
#include "gmp.h"

#include "crypto.h"
#include "rsa.h"
#include "mkrsa.h"


/* ==================================================================

   Flow of the Secure Multimedia Layer protocol
   
   ================================================================== */


/*
Server              Client
-------------------------------
Create              Create

LoadKeyPair         LoadKeyPair
Config

                    Connect
Accept

Encrypt
                    Decrypt
Encrypt
                    Decrypt


Destroy             Destroy
-------------------------------
*/




/* ==================================================================

   Definitions
   
   ================================================================== */

#define FALSE 0
#define TRUE 1


#define INADDR_NONE -1
#define SOCKET_ERROR -1
#define INVALID_SOCKET -1

#define SML_SERVER_PORT 9001
#define SML_PROXY_PORT 9080

#define SML_DEFAULT_ECP_I 5
#define SML_DEFAULT_ECP_P 10
#define SML_DEFAULT_ECP_B 1
#define SML_DEFAULT_ECP_PKT_SIZE 1400

#define SML_RPS_MULTI_KEY_RSA 0

#define SML_SUCCESS                    0
#define SML_ERROR_NO_MEMORY            1
#define SML_ERROR_NULL_SESSION         2
#define SML_ERROR_NULL_KEY             3
#define SML_ERROR_ALREADY_CONNECTED    4
#define SML_ERROR_CRYPTO_KEY_UNDEFINED 5
#define SML_ERROR_KEY_TYPE_UNSUPPORT   6
#define SML_ERROR_FILE_ERROR           7
#define SML_ERROR_SOCKET_ERROR         8
#define SML_ERROR_INTEGRITY_FAIL       9
#define SML_ERROR_NO_SUCH_USER         10
#define SML_ERROR_INCORRECT_PASSWD     11
#define SML_ERROR_CRYPT_ERROR          12
#define SML_ERROR_NEW_KEY_ERROR        13



#define MAX_USER_STRING_LEN 256
#define MAX_PASSWD_STRING_LEN 256
#define MAX_KEY_STRING_LEN 4096


///////////////////////////////////////////////////////////////////
// struct that holds all information uniquly identify a SML session
#define SESSION_NUM_KEY 4
enum {MY_PUBLIC_KEY, MY_PRIVATE_KEY, PUBLIC_KEY, RPS_KEY};

typedef struct {

    // communication socket
    int last_bind_port; // server tcp port
	int servSd;         // server tcp socket
    int socket;         // client tcp socket

    // states and flags
    unsigned int seq;
    unsigned char anonymous_mode;

    // public keys and private keys
    CRYPTO_KEY key[SESSION_NUM_KEY];
    char *__signature, *__client_signature;
    int __signature_len, __client_signature_len;

    // ECP and RPS
    unsigned int I, P, B, pkt_size;
    char *__reenc_key, *__dec_key, *__dec_key2;
    int __reenc_key_len, __dec_key_len, __dec_key2_len;

    // pre-calcuated ECP values for faster encryption/decryption
	unsigned int __cipher_unit_size_mp, __cipher_unit_size;
    unsigned int __plaintext_unit_size_mp, __plaintext_unit_size;
    unsigned int __encrypt_unit_total;
   	unsigned int __encrypt_unit_per_block, __encrypt_unit_last_block;
    unsigned int __plaintext_pkt_size, __cipher_pkt_size;
	unsigned int __unencrypt_size_last_block;
	unsigned int __unencrypt_size_per_block;
    char *__cipher_buffer;
    char *__plaintext_buffer;

} SML_SESSION;

typedef struct {
    unsigned int I, P, B, pkt_size;
    char *__reenc_key, *__dec_key, *__dec_key2;
    int __reenc_key_len, __dec_key_len, __dec_key2_len;
} RPS;

typedef struct {
	CRYPTO_KEY public_key;
	char *signed_public_key;
	int signed_public_key_len;
} SML_SIGNATURE;


/* ==================================================================

   Prototypes of the SML API
   
   ================================================================== */

int __TCP_ReadFully (int socket, char *data, int length);

void SML_InitSession(SML_SESSION* session);
void SML_DestroySession(SML_SESSION* session);

int SML_LoadKeyPair(SML_SESSION* session, char keyfilename[], char passwd[], unsigned int key_type);
int SML_SaveKeyPair(SML_SESSION* session, char keyfilename[], char passwd[]);
int SML_NewKeyPair(SML_SESSION* session, unsigned int bit_length, unsigned int pub_exp, unsigned int key_type);

/* specific for proxy */
void SML_InitSignature(SML_SIGNATURE* signature);
void SML_DestroySignature(SML_SIGNATURE* signature);
void SML_GetClientSignature(SML_SESSION* session, SML_SIGNATURE* signature);
void SML_SetClientSignature(SML_SESSION* session, SML_SIGNATURE signature);

int SML_Connect(SML_SESSION* session, char host[], int port);
SML_SESSION* SML_Accept(SML_SESSION* session, int port, char list_filename[]);

void SML_InitRps(SML_SESSION* session, int I, int P, int B, int pkt_size, int algorithm, int bit_len);
void SML_SaveRpsSetting(SML_SESSION* session, char filename[], char passwd[]);
void SML_LoadRpsSetting(SML_SESSION* session, char filename[], char passwd[]);
void SML_NewRpsReEncryptionKey(SML_SESSION *session);
void SML_SendRpsSetting(SML_SESSION* session);
void SML_ReceiveRpsSetting(SML_SESSION* session);
int SML_GetRpsEncryptedPacketSize(SML_SESSION* session);

void RPS_Init(RPS* rps);
void RPS_Destroy(RPS* rps);
void SML_GetRpsSetting(SML_SESSION* session, RPS* rps);
void SML_SetRpsSetting(SML_SESSION* session, RPS rps);

int SML_GetRpsEncryptedPacketSize(SML_SESSION* session);

int SML_TcpSendEncryptRps(SML_SESSION* session, char data[], unsigned int len);
int SML_TcpReceiveDecryptRps(SML_SESSION* session, char data[], unsigned int len);

int SML_TcpSendReEncryptRps(SML_SESSION* session, char data[], unsigned int len);
int SML_TcpReceiveRps(SML_SESSION* session, char data[], unsigned int len);
