
#include <stdio.h>
#include <stdlib.h>

#include "rsa.h"



gmp_randstate_t rand_state;
int randstate_initialized = 0;

/*
 * __RSA_GenerateRandomParameters()
 *
 * generate random RSA key parameters p, q, and phi
 *
 */
void __RSA_GenerateRandomParameters(unsigned int bit_length, mpz_t mod, mpz_t phi)
{
    mpz_t p, q, temp;

    // initialize mpz_t variables
    mpz_init2(p, bit_length);
    mpz_init2(q, bit_length);
    mpz_init2(temp, bit_length);

    // initialize random number generator
    //
    // !!!!! time randomness is bad, need to be changed !!!!!!
    //
	if (!randstate_initialized) {
		randstate_initialized = 1;
   	 	gmp_randinit_default(rand_state);
   		gmp_randseed_ui(rand_state, time(NULL));
	}

    mpz_ui_pow_ui(temp, 2, bit_length-1);
    mpz_sub_ui(temp, temp, 1);
    
    // find random p & q s.t. mod = p*q is at least bit_length-1 bits
    do {
        mpz_urandomb(p, rand_state, bit_length/2);
        mpz_urandomb(q, rand_state, bit_length/2);
        mpz_mul(mod, p, q);
    } while (mpz_cmp(temp, mod) >= 0);
    
    do { // inc. p until probably prime
        mpz_nextprime(p, p);
    } while (mpz_probab_prime_p(p, 20) == 0);
    
    do { // inc. q until probably prime
        mpz_nextprime(q, q);
    } while (mpz_probab_prime_p(q, 20) == 0);
    
    // compute output parameter mod & phi
    mpz_mul(mod, p, q);
    mpz_sub_ui(p, p, 1);
    mpz_sub_ui(q, q, 1);
    mpz_mul(phi, p, q);
    
    // release temporary resources
    mpz_clear(p);
    mpz_clear(q);
    mpz_clear(temp);
}

/*
 * __RSA_GeneratePrivateExp()
 *
 * generate a private exponent given the public exponent and phi
 *
 */
void __RSA_GeneratePrivateExp(unsigned int bit_length, mpz_t pri_exp, mpz_t pub_exp, mpz_t phi)
{
    mpz_t temp1, temp2, temp3, temp4, left, right;
    mpz_t a, b, temp;
 
    mpz_init2(a,     bit_length);
    mpz_init2(b,     bit_length); 
    mpz_init2(temp,  bit_length);
    mpz_init2(temp1, bit_length);
    mpz_init2(temp2, bit_length);
    mpz_init2(temp3, bit_length);
    mpz_init2(temp4, bit_length);
    mpz_init2(left,  bit_length);
    mpz_init2(right, bit_length);

    mpz_set(a, pub_exp);
    mpz_set(b, phi);
    
    mpz_set_ui(temp1, 1);
    mpz_set_ui(temp2, 0);
    mpz_set_ui(temp3, 0);
    mpz_set_ui(temp4, 1);

    // find priKey such that pubKey * priKey = 1 (mod phi)
    while (1) {
		mpz_fdiv_q(right, b, a);
        mpz_mul(temp, right, a);
        mpz_sub(b, b, temp);
        mpz_mul(temp, right, temp1);
        mpz_sub(temp3, temp3, temp);
        mpz_mul(temp, right, temp2);
        mpz_sub(temp4, temp4, temp);
        if (mpz_cmp_si(b, 1) == 0)
            break;

        mpz_fdiv_q(left, a, b);
        mpz_mul(temp, left, b);
        mpz_sub(a, a, temp);
        mpz_mul(temp, left, temp3);
        mpz_sub(temp1, temp1, temp);
        mpz_mul(temp, left, temp4);
        mpz_sub(temp2, temp2, temp);
        if (mpz_cmp_si(a, 1) == 0)
            break;
    }

    // compute output value pri_exp
    if (mpz_cmp_si(a, 1) == 0) {
		if (mpz_sgn(temp1) < 0) 
            mpz_addmul(temp1, temp2, phi);
        mpz_set(pri_exp, temp1);
    }
    else {
    	if (mpz_sgn(temp3) < 0)
    		mpz_addmul(temp3, temp4, phi);
        mpz_set(pri_exp, temp3);
    }
    
    // release temporary resource
    mpz_clear(temp);
    mpz_clear(temp1);
    mpz_clear(temp2);
    mpz_clear(temp3);
    mpz_clear(temp4);
    mpz_clear(a);
    mpz_clear(b);
    mpz_clear(left);
    mpz_clear(right);
}

/*
 * RSA_NewKeyPair()
 *
 * generate a new key pair with specific public exponent
 * (use default public exponent if parameter pub_exp_int is zero)
 *
 */
BOOL RSA_NewKeyPair(RSA_KEY_PAIR *key_pair, unsigned int bit_length, unsigned long pub_exp_int)
{
	int i;
    mpz_t temp, pub_exp, pri_exp, mod, phi;
    
    // don't allow too small bit length
    if (bit_length < 64)
        bit_length = 64;
    
    mpz_init2(temp,    bit_length);
    mpz_init2(pub_exp, bit_length);
    mpz_init2(pri_exp, bit_length);
    mpz_init2(mod,     bit_length);
    mpz_init2(phi,     bit_length);
    
    // generate random parameters mod & phi
    __RSA_GenerateRandomParameters(bit_length, mod, phi);
    
    // set the public exponent, must be odd number
    if (pub_exp_int == RSA_RANDOM_PUBLIC_EXPONENT) {
        mpz_urandomb(pub_exp, rand_state, bit_length/2);
		if (mpz_even_p(pub_exp)) {
			mpz_add_ui(pub_exp, pub_exp, 1);
		}
	}
    else {
		if (pub_exp_int % 2 == 0)
       		pub_exp_int++;
	    mpz_set_ui(pub_exp, pub_exp_int);
	}
    
    // increase public exponent by 2 until pub_exp and phi are relative primes
    mpz_gcd(temp, pub_exp, phi);
    while (mpz_cmp_ui(temp, 1) != 0) { 
        mpz_add_ui(pub_exp, pub_exp, 2);
        mpz_gcd(temp, pub_exp, phi);
    }
    
    // generate random private exponent
    __RSA_GeneratePrivateExp(bit_length, pri_exp, pub_exp, phi);


	//////////////////////////////////////////////////////////////////////////
    // create the keys with generated values
	//////////////////////////////////////////////////////////////////////////

	if (key_pair->public_key.param) {
		for (i=0; i<key_pair->public_key.num_param; i++) {
			mpz_clear(key_pair->public_key.param[i]);
		}
	}

	if (key_pair->private_key.param) {
		for (i=0; i<key_pair->private_key.num_param; i++) {
			mpz_clear(key_pair->private_key.param[i]);
		}
	}

	// reallocate to new size
	if ((key_pair->public_key.param = (mpz_t*)realloc(key_pair->public_key.param, sizeof(mpz_t)*2)) == NULL)
		goto error;

	mpz_init2(key_pair->public_key.param[0], bit_length);
	mpz_init2(key_pair->public_key.param[1], bit_length);

	key_pair->public_key.type = CRYPTO_KEY_RSA;
	key_pair->public_key.num_param = RSA_NUM_PARAM;

	if ((key_pair->private_key.param = (mpz_t*)realloc(key_pair->private_key.param, sizeof(mpz_t)*2)) == NULL)
		goto error;

	mpz_init2(key_pair->private_key.param[0], bit_length);
	mpz_init2(key_pair->private_key.param[1], bit_length);

	key_pair->private_key.type = CRYPTO_KEY_RSA;
	key_pair->private_key.num_param = RSA_NUM_PARAM;

	// store key parameters
	key_pair->type = CRYPTO_KEY_RSA;
	
    mpz_set(key_pair->public_key.param[RSA_PARAM_EXPONENT], pub_exp);
    mpz_set(key_pair->public_key.param[RSA_PARAM_MODULUS], mod);
    
    mpz_set(key_pair->private_key.param[RSA_PARAM_EXPONENT], pri_exp);
    mpz_set(key_pair->private_key.param[RSA_PARAM_MODULUS], mod);

	// release temporary resources
    mpz_clear(temp);
    mpz_clear(pub_exp);
    mpz_clear(pri_exp);
    mpz_clear(mod);
    mpz_clear(phi);
    
    return TRUE;
    
error:

	if (key_pair->public_key.param) {
		for (i=0; i<key_pair->public_key.num_param; i++) {
			mpz_clear(key_pair->public_key.param[i]);
		}
		free(key_pair->public_key.param);
		
		key_pair->public_key.type = CRYPTO_KEY_UNDEFINED;
		key_pair->public_key.num_param = 0;
		key_pair->public_key.param = NULL;
	}

	if (key_pair->private_key.param) {
		for (i=0; i<key_pair->private_key.num_param; i++) {
			mpz_clear(key_pair->private_key.param[i]);
		}
		free(key_pair->private_key.param);

		key_pair->private_key.type = CRYPTO_KEY_UNDEFINED;
		key_pair->private_key.num_param = 0;
		key_pair->private_key.param = NULL;
	}

    mpz_clear(temp);
    mpz_clear(pub_exp);
    mpz_clear(pri_exp);
    mpz_clear(mod);
    mpz_clear(phi);
	
	return FALSE;
}



/*
 * RSA_LoadKeyPair()
 *
 * 
 *
 */
BOOL RSA_LoadKeyPair(RSA_KEY_PAIR *key_pair, char key_filename[], char passwd[])
{
    FILE *fp = NULL;
    unsigned int type;
    char *temp = NULL;
    int i;
    
    // open key file
    if ((fp = fopen(key_filename, "rt")) == NULL) {
        goto error_load_key_pair;
    }
    
    // allocate memory for temporary string
    if ((temp = (char*)malloc(sizeof(char) * RSA_MAX_KEY_STRING_LEN)) == NULL) {
        goto error_load_key_pair;
    }
    
    // release previous allocation
	if (key_pair->public_key.param) {
		for (i=0; i<key_pair->public_key.num_param; i++) {
			mpz_clear(key_pair->public_key.param[i]);
		}
	}

	if (key_pair->private_key.param) {
		for (i=0; i<key_pair->private_key.num_param; i++) {
			mpz_clear(key_pair->private_key.param[i]);
		}
	}    

	// reallocate to the new size
	if ((key_pair->public_key.param = (mpz_t*)realloc(key_pair->public_key.param, sizeof(mpz_t)*2)) == NULL) {
        goto error_load_key_pair;
    }

	for (i=0; i<2; i++) {
		mpz_init(key_pair->public_key.param[i]);
	}

	if ((key_pair->private_key.param = (mpz_t*)realloc(key_pair->private_key.param, sizeof(mpz_t)*2)) == NULL) {
        goto error_load_key_pair;
    }

	for (i=0; i<2; i++) {
		mpz_init(key_pair->private_key.param[i]);
	}
	
	key_pair->type = CRYPTO_KEY_RSA;
	key_pair->public_key.type = CRYPTO_KEY_RSA;
	key_pair->public_key.num_param = RSA_NUM_PARAM;
	key_pair->private_key.type = CRYPTO_KEY_RSA;
	key_pair->private_key.num_param = RSA_NUM_PARAM;
    
    // load public key from file
    fgets(temp, RSA_MAX_KEY_STRING_LEN, fp);
    if (temp[strlen(temp)-1] == '\n')
        temp[strlen(temp)-1] = 0;
    mpz_set_str(key_pair->public_key.param[RSA_PARAM_EXPONENT], temp, 10);
   
    fgets(temp, RSA_MAX_KEY_STRING_LEN, fp);
    if (temp[strlen(temp)-1] == '\n')
        temp[strlen(temp)-1] = 0;
    mpz_set_str(key_pair->public_key.param[RSA_PARAM_MODULUS], temp, 10);

    // load private key from file    
    fgets(temp, RSA_MAX_KEY_STRING_LEN, fp);
    if (temp[strlen(temp)-1] == '\n')
        temp[strlen(temp)-1] = 0;
    mpz_set_str(key_pair->private_key.param[RSA_PARAM_EXPONENT], temp, 10);

    fgets(temp, RSA_MAX_KEY_STRING_LEN, fp);
    if (temp[strlen(temp)-1] == '\n')
        temp[strlen(temp)-1] = 0;
    mpz_set_str(key_pair->private_key.param[RSA_PARAM_MODULUS], temp, 10);
    
    free(temp);
    fclose(fp);
    
    return TRUE;
    
error_load_key_pair:

	if (key_pair->public_key.param != NULL) {
		mpz_clear(key_pair->public_key.param[0]);
		mpz_clear(key_pair->public_key.param[1]);
	}
	free(key_pair->public_key.param);

	key_pair->public_key.type = CRYPTO_KEY_UNDEFINED;
	key_pair->public_key.num_param = 0;
	key_pair->public_key.param = NULL;

	if (key_pair->private_key.param != NULL) {
		mpz_clear(key_pair->private_key.param[0]);
		mpz_clear(key_pair->private_key.param[1]);
	}
	free(key_pair->private_key.param);

	key_pair->private_key.type = CRYPTO_KEY_UNDEFINED;
	key_pair->private_key.num_param = 0;
	key_pair->private_key.param = NULL;
	
	if (temp)
		free(temp);
	
	if (fp)
		fclose(fp);
	
	return FALSE;
}

/*
 * RSA_SaveKeyPair()
 *
 * 
 *
 */
BOOL RSA_SaveKeyPair(RSA_KEY_PAIR key_pair, char key_filename[], char passwd[])
{
    FILE *fp = NULL;
    char *temp = NULL;
    int i;
    
    // open key file
    if ((fp = fopen(key_filename, "wt")) == NULL) {
		goto error_save_key_pair;
	}
    
    // allocate memory for temporary string
    if ((temp = (char*)malloc(sizeof(char) * RSA_MAX_KEY_STRING_LEN)) == NULL) {
    	goto error_save_key_pair;
    }

    // save public key to file
    mpz_get_str(temp, 10, key_pair.public_key.param[RSA_PARAM_EXPONENT]);
    fprintf(fp, "%s\n", temp);
    mpz_get_str(temp, 10, key_pair.public_key.param[RSA_PARAM_MODULUS]);
    fprintf(fp, "%s\n", temp);

    // save private key to file
    mpz_get_str(temp, 10, key_pair.private_key.param[RSA_PARAM_EXPONENT]);
    fprintf(fp, "%s\n", temp);
    mpz_get_str(temp, 10, key_pair.private_key.param[RSA_PARAM_MODULUS]);
    fprintf(fp, "%s\n", temp);
    
    return TRUE;

error_save_key_pair:

	if (temp)
    	free(temp);

   	if (fp)
    	fclose(fp);
    	
    return FALSE;
}





/*
 * __RSA_Encrypt()
 *
 * perform RSA encryption on a mpz_t value
 *
 */
void __RSA_Encrypt(mpz_t cipher, mpz_t data, RSA_KEY key)
{
    __mpz_struct *exp = key.param[RSA_PARAM_EXPONENT];
    __mpz_struct *mod = key.param[RSA_PARAM_MODULUS];
    
    mpz_powm(cipher, data, exp, mod);
}

/*
 * __RSA_Decrypt()
 *
 * perform RSA decryption on a mpz_t value
 *
 */
void __RSA_Decrypt(mpz_t data, mpz_t cipher, RSA_KEY key)
{
    __mpz_struct *exp = key.param[RSA_PARAM_EXPONENT];
    __mpz_struct *mod = key.param[RSA_PARAM_MODULUS];
    
    mpz_powm(cipher, data, exp, mod);
}

/*
 * __RSA_PublicEncrypt()
 *
 * perform RSA encryption using the public key
 *
 */
void __RSA_PublicEncrypt(mpz_t cipher, mpz_t data, RSA_KEY_PAIR key_pair)
{
	__RSA_Encrypt(cipher, data, key_pair.public_key);
}

/*
 * __RSA_PublicDecrypt()
 *
 * perform RSA decryption using the public key
 *
 */
void __RSA_PublicDecrypt(mpz_t data, mpz_t cipher, RSA_KEY_PAIR key_pair)
{
	__RSA_Decrypt(data, cipher, key_pair.public_key);
}

/*
 * __RSA_PrivateEncrypt()
 *
 * perform RSA encryption using the private key
 *
 */
void __RSA_PrivateEncrypt(mpz_t cipher, mpz_t data, RSA_KEY_PAIR key_pair)
{
	__RSA_Encrypt(cipher, data, key_pair.private_key);
}

/*
 * __RSA_PrivateDecrypt()
 *
 * perform RSA decryption using the private key
 *
 */
void __RSA_PrivateDecrypt(mpz_t data, mpz_t cipher, RSA_KEY_PAIR key_pair)
{
	__RSA_Decrypt(data, cipher, key_pair.private_key);
}







/*
 * RSA_Encrypt()
 *
 * perform RSA encryption on byte array
 *
 */
BOOL RSA_Encrypt(char cipher[], int *out_len, char data[], int in_len, RSA_KEY key)
{
	int i, in_offs, out_offs;
	int block_len_mp, block_len_byte, min_out_len, GMP_LIMB_BYTES = mp_bits_per_limb  / 8;
	mpz_t mpz_data;
	
	// compute the encrypted data size
	block_len_mp = key.param[RSA_PARAM_MODULUS]->_mp_size - 1;
	block_len_byte = block_len_mp * GMP_LIMB_BYTES;
	
	//fprintf(stderr, "[DEBUG] GMP_LIMB_BITS = %d\n", mp_bits_per_limb);

    min_out_len = ((in_len+block_len_byte-1) / block_len_byte) * (block_len_byte + GMP_LIMB_BYTES);

    //fprintf(stderr, "[DEBUG] in_len = %d, min_out_len = %d\n", in_len, min_out_len);

    // report the minimum required output buffer size if not enough
    if (*out_len < min_out_len) {
    	*out_len = min_out_len;
    	return FALSE;
    }
	*out_len = min_out_len;

    // perform encryption on block level
    mpz_init2(mpz_data, key.param[RSA_PARAM_MODULUS]->_mp_size * mp_bits_per_limb);
    memset(cipher, 0, *out_len);
    in_offs = out_offs = 0;

	while (in_offs + block_len_byte <= in_len) {
		mpz_data->_mp_size = block_len_mp;
        memcpy(mpz_data->_mp_d, data+in_offs, block_len_byte);
        in_offs += block_len_byte;

        mpz_powm(mpz_data, mpz_data, key.param[RSA_PARAM_EXPONENT], key.param[RSA_PARAM_MODULUS]);
        
        memcpy(cipher+out_offs, mpz_data->_mp_d, mpz_data->_mp_size * GMP_LIMB_BYTES);
        out_offs += block_len_byte + GMP_LIMB_BYTES;
    }
    
    // last block, input need padding
    if (in_offs < in_len) {
		mpz_data->_mp_size = block_len_mp;
		memset(mpz_data->_mp_d, 0, block_len_byte);
	    memcpy(mpz_data->_mp_d, data+in_offs, in_len - in_offs);
	    in_offs += block_len_byte;
	
	    mpz_powm(mpz_data, mpz_data, key.param[RSA_PARAM_EXPONENT], key.param[RSA_PARAM_MODULUS]);
	    
	    memcpy(cipher+out_offs, mpz_data->_mp_d, mpz_data->_mp_size * GMP_LIMB_BYTES);
	    out_offs += block_len_byte + GMP_LIMB_BYTES;
    }
    
    mpz_clear(mpz_data);
    
    return TRUE;
}

/*
 * RSA_Decrypt()
 *
 * perform RSA decryption on byte array
 *
 */
BOOL RSA_Decrypt(char data[], int *out_len, char cipher[], int in_len, RSA_KEY key)
{
	int i, in_offs, out_offs;
	int block_len_mp, block_len_byte, min_out_len, GMP_LIMB_BYTES = mp_bits_per_limb  / 8;
	mpz_t mpz_data;

	// compute the decrypted data size
	block_len_mp = key.param[RSA_PARAM_MODULUS]->_mp_size;
	block_len_byte = block_len_mp * GMP_LIMB_BYTES;
	
	//fprintf(stderr, "[DEBUG] GMP_LIMB_BITS = %d\n", mp_bits_per_limb );

    min_out_len = ((in_len-block_len_byte-1) / block_len_byte) * (block_len_byte - GMP_LIMB_BYTES);

    //fprintf(stderr, "[DEBUG] in_len = %d, min_out_len = %d\n", in_len, min_out_len);

    // report the minimum required output buffer size if not enough
    if (*out_len < min_out_len) {
    	*out_len = min_out_len;
    	return FALSE;
    }
	*out_len = min_out_len;

    // perform decryption on block level
    mpz_init2(mpz_data, key.param[RSA_PARAM_MODULUS]->_mp_size * mp_bits_per_limb);
    memset(data, 0, *out_len);
    in_offs = out_offs = 0;

	while (in_offs + block_len_byte <= in_len) {
		mpz_data->_mp_size = block_len_mp;
        memcpy(mpz_data->_mp_d, cipher+in_offs, block_len_byte);
        in_offs += block_len_byte;

        mpz_powm(mpz_data, mpz_data, key.param[RSA_PARAM_EXPONENT], key.param[RSA_PARAM_MODULUS]);
        
        memcpy(data+out_offs, mpz_data->_mp_d, mpz_data->_mp_size * GMP_LIMB_BYTES);
        out_offs += block_len_byte - GMP_LIMB_BYTES;
    }
    
    // last block, input need padding
    if (in_offs < in_len) {
		mpz_data->_mp_size = block_len_mp;
		memset(mpz_data->_mp_d, 0, block_len_byte);
        memcpy(mpz_data->_mp_d, cipher+in_offs, in_len-in_offs);
        in_offs += block_len_byte;

        mpz_powm(mpz_data, mpz_data, key.param[RSA_PARAM_EXPONENT], key.param[RSA_PARAM_MODULUS]);
        
        memcpy(data+out_offs, mpz_data->_mp_d, mpz_data->_mp_size * GMP_LIMB_BYTES);
        out_offs += block_len_byte - GMP_LIMB_BYTES;
    }
    
    mpz_clear(mpz_data);
    return TRUE;
}

/*
 * RSA_PublicEncrypt()
 *
 * perform RSA encryption on byte array using public key
 *
 */
BOOL RSA_PublicEncrypt(char cipher[], int *out_len, char data[], int in_len, RSA_KEY_PAIR key_pair)
{
	return RSA_Encrypt(cipher, out_len, data, in_len, key_pair.public_key);
}

/*
 * RSA_PublicDecrypt()
 *
 * perform RSA decryption on byte array using public key
 *
 */
BOOL RSA_PublicDecrypt(char data[], int *out_len, char cipher[], int in_len, RSA_KEY_PAIR key_pair)
{
	return RSA_Decrypt(data, out_len, cipher, in_len, key_pair.public_key);
}

/*
 * RSA_PrivateEncrypt()
 *
 * perform RSA encryption on byte array using private key
 *
 */
BOOL RSA_PrivateEncrypt(char cipher[], int *out_len, char data[], int in_len, RSA_KEY_PAIR key_pair)
{
	return RSA_Encrypt(cipher, out_len, data, in_len, key_pair.private_key);
}

/*
 * RSA_PrivateDecrypt()
 *
 * perform RSA decryption on byte array using private key
 *
 */
BOOL RSA_PrivateDecrypt(char data[], int *out_len, char cipher[], int in_len, RSA_KEY_PAIR key_pair)
{
	return RSA_Decrypt(data, out_len, cipher, in_len, key_pair.private_key);
}

/*
 * RSA_SignPublicKey()
 *
 * encrypt the public key using the corresponding private key, output as string
 *
 */
BOOL RSA_SignPublicKey(char output[], int *out_len, RSA_KEY_PAIR pair)
{
	int result;
	int len;
	char *str = NULL;

	len = CRYPTO_KeyToStringAlloc(&str, pair.public_key);

	result = RSA_Encrypt(output, out_len, str, len, pair.private_key);

	if (str)
		free(str);

	return result;
}

/*
 * RSA_CheckSignedPublicKey()
 *
 * decrypt the cipher using a public key and then check the plaintext against that public key
 *
 */
BOOL RSA_CheckSignedPublicKey(char cipher[], int in_len, RSA_KEY public_key)
{
	int i;
	unsigned int len = in_len;
	char *str = NULL;
	RSA_KEY decrypted_key;

	// decrypt the private-encrypted key
	if ((str = (char*)malloc(sizeof(char) * in_len)) == NULL)
		goto check_signed_public_key_error;

	RSA_Decrypt(str, &len, cipher, in_len, public_key);

	// compare the decrypted key against the public key
	CRYPTO_InitKey(&decrypted_key);

	if (!CRYPTO_ParseKeyFromString(&decrypted_key, str))
		goto check_signed_public_key_error;

	if (decrypted_key.type != CRYPTO_KEY_RSA) // check key type
		goto check_signed_public_key_error;

	if (decrypted_key.num_param != public_key.num_param) // check no. of parameters
		goto check_signed_public_key_error;

	for (i=0; i<public_key.num_param; i++) { // check each parameter
		if (mpz_cmp(decrypted_key.param[i], public_key.param[i]) != 0) {
			goto check_signed_public_key_error;
		}
	}
	
	// release temporary resources
	if (str)
		free(str);
		
	CRYPTO_DestroyKey(&decrypted_key);

	return TRUE;

check_signed_public_key_error:
	
	if (str)
		free(str);
		
	CRYPTO_DestroyKey(&decrypted_key);

	return FALSE;
}

unsigned int RSA_GetPlaintextUnitSize(RSA_KEY key)
{
	return (key.param[RSA_PARAM_MODULUS]->_mp_size - 1) * (mp_bits_per_limb / 8);
}

unsigned int RSA_GetCipherUnitSize(RSA_KEY key)
{
	return key.param[RSA_PARAM_MODULUS]->_mp_size * (mp_bits_per_limb / 8);
}
