#include <iostream>                                                                                                                                          
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <math.h>
#include <string.h>

#ifndef EDITOR
#define EDITOR

#define SEPARATOR 0
#define INSERT    1
#define DELETE    2
#define SPLIT     3
#define REPLACE   4
#define MAX_BUFFER_SIZE 2000

class PieceTableNode{  // It is a doubly linked list.
  public:
  
  int start, length, replaced_start;
  PieceTableNode *pNext, *pPrev;
  
  public:
  
  PieceTableNode(int p,int len, int rs);
  PieceTableNode(PieceTableNode *val);
  void SetValue(int p,int len, int rs);
};

class PieceTable{
  public:
  
  PieceTableNode *head, *tail;
  
  public:
         
  PieceTable();
  ~PieceTable();
  void FreePieceTable();

  void search(int loci, PieceTableNode **node, int *offset);
  void search_pos(int position, PieceTableNode **node, int *offset);
  void move(PieceTableNode *from, int offset1, int steps, PieceTableNode **to, int *offset2);
  bool split(PieceTableNode *node, int offset, PieceTableNode **new_node);
  
  void print();
};

class EditHistoryNode{  // It is a stack.
  public:
  
  int type, loci, start, length, replaced_start;
  EditHistoryNode *pNext;

  public:
  
  EditHistoryNode(int t, int lo, PieceTableNode *val, EditHistoryNode *next);
};

class EditHistory{
  public:
  
  EditHistoryNode *head;
  
  public:
  
  EditHistory();
  ~EditHistory();
  void FreeEditHistory();
  void Push(int t, int lo, PieceTableNode *val);
  void Pop();
};

class TextEditor{
  public:
  
  EditHistory *edit_history;  // A stack to save the editing history
  PieceTable *sequence;       // A linked list to represent the content in the text editor
  PieceTable *reference;      // A linked list to represent the reference sequence
  
  int original_length, added_length;
  char *original, *buffer;
  
  public:
  
  TextEditor();
  ~TextEditor();
  void Reset();
  void SetOriginal(char *seq);
  
  void insert_break_in_history();
  void insert_n(int position, int n, char *add);
  void remove_n(int position, int n, bool bText, char *removed_text);
  void replace_blank_n(int position, int n, bool bText, char *replaced_text);
  void replace_n(int position, int n, char *add, bool bText, char *replaced_text);
  void undo();

  void get_text_from_loci(char *seq, int loci, int n);
  void get_text(PieceTable *piecetable, char *seq);
  void get_original_empty(char *seq);
  void get_loci_seq(int *loci_seq, int *pos_seq);
  void testprint(char *seq);
};

/*******************************************************
                   PieceTableNode
 *******************************************************/

PieceTableNode::PieceTableNode(int p,int len, int rs){
  SetValue(p,len,rs);
  pNext = pPrev = NULL;
}

PieceTableNode::PieceTableNode(PieceTableNode *value){
  start = value->start;
  length = value->length;
  replaced_start = value->replaced_start;
  pNext = pPrev = NULL;
}

void PieceTableNode::SetValue(int p,int len, int rs){
  start = p;
  length = len;
  replaced_start = rs;
}

void LinkPieceTableNode(PieceTableNode *p, PieceTableNode *n){
  if(p!=NULL)
    p->pNext = n;
  if(n!=NULL)
    n->pPrev = p;
}

void JoinPieceTableNode(PieceTableNode *p, PieceTableNode *c, PieceTableNode *n){
  LinkPieceTableNode(p,c);
  LinkPieceTableNode(c,n);
}

PieceTable::PieceTable(){
  head = new PieceTableNode(-1,0,-1);
  tail = new PieceTableNode(-1,0,-1);
  LinkPieceTableNode(head,tail);
}

PieceTable::~PieceTable(){
  FreePieceTable();
  delete head;
  delete tail;
}

void PieceTable::FreePieceTable(){
  PieceTableNode *current, *temp;
  for(current=head->pNext;current!=tail;){
    temp = current;
    current=current->pNext;
    delete temp;
  }
  LinkPieceTableNode(head,tail);
}

// Return the node where loci is located.
void PieceTable::search(int loci, PieceTableNode **node, int *offset){
  PieceTableNode *current;

  for(current=head->pNext;current!=tail;current=current->pNext){
    if((current->start <= loci)&&(loci <= current->start+current->length-1)){
      *node = current;
      *offset = loci-current->start;
      return;
    }
  }
  *node = tail;
  *offset = 0;
  return;
}

// Return the node where loci is located.
void PieceTable::search_pos(int position, PieceTableNode **node, int *offset){
  int cum_len=0;
  PieceTableNode *current;

  for(current=head->pNext;current!=tail;current=current->pNext){
    if(cum_len+current->length > position){
      *node = current;
      *offset = position-cum_len;
      return;
    }
    cum_len += current->length;
  }
  *node = tail;
  *offset = 0;
  return;
}

void PieceTable::move(PieceTableNode *from, int offset1, int steps, PieceTableNode **to, int *offset2){
  int total_steps,cum_len=0;
  PieceTableNode *current;
  
  total_steps = offset1+steps;
  for(current=from;current!=tail;current=current->pNext){
    if(cum_len+current->length > total_steps){
      *to = current;
      *offset2 = total_steps-cum_len;
      return;
    }
    cum_len += current->length;
  }
  *to = tail;
  *offset2 = 0;
  return;
}

bool PieceTable::split(PieceTableNode *node, int offset, PieceTableNode **new_node){
  if(offset==0){
    *new_node = node;
    return false;
  }else{
    *new_node = new PieceTableNode(node->start+offset,node->length-offset,node->replaced_start+offset);
    node->length = offset;
    JoinPieceTableNode(node,*new_node,node->pNext);
    return true;
  }
}

void PieceTable::print(){
  PieceTableNode *current;
  for(current=head;current!=tail;current=current->pNext)
    printf("Start = %d, Length = %d\n",current->start,current->length);
  printf("\n");
}

/*******************************************************
                   EditHistory
 *******************************************************/

EditHistoryNode::EditHistoryNode(int t, int lo, PieceTableNode *val, EditHistoryNode *next){
  type = t;
  loci = lo;
  if(val!=NULL){
    start = val->start;
    length = val->length;
    replaced_start = val->replaced_start;
  }
  pNext = next;
}

EditHistory::EditHistory(){
  head = NULL;
}

EditHistory::~EditHistory(){
  FreeEditHistory();
}

void EditHistory::FreeEditHistory(){
  EditHistoryNode *current, *temp;
  for(current=head;current!=NULL;){
    temp = current;
    current=current->pNext;
    delete temp;
  }
}

void EditHistory::Push(int t, int lo, PieceTableNode *val){
  head = new EditHistoryNode(t,lo,val,head);
}

void EditHistory::Pop(){
  EditHistoryNode *temp;
  
  temp = head;
  head = head->pNext;
  delete temp;
}

/*******************************************************
                  TextEditor
 *******************************************************/

TextEditor::TextEditor(){
  buffer = (char*)malloc((unsigned int)MAX_BUFFER_SIZE*sizeof(char));
  sequence = new PieceTable();
  reference = new PieceTable();
  edit_history = new EditHistory();
}

TextEditor::~TextEditor(){
  free(buffer);
  delete sequence;
  delete reference;
  delete edit_history;
}

void TextEditor::Reset(){
  sequence->FreePieceTable();
  reference->FreePieceTable();
  edit_history->FreeEditHistory();
}

void TextEditor::SetOriginal(char *seq){
  original = seq;
  original_length = strlen(seq);
  added_length = 0;
  buffer[0]='\0';
  JoinPieceTableNode(sequence->head, new PieceTableNode(0, original_length, 0), sequence->tail);
  JoinPieceTableNode(reference->head, new PieceTableNode(0, original_length, 0), reference->tail);
}

void TextEditor::insert_n(int position, int n, char *add){
  int offset, loci_add;
  PieceTableNode *node, *new_node, *ins_node;
  
  loci_add = original_length+added_length;
  strncat(buffer,add,n);

  sequence->search_pos(position,&node,&offset);
  if(sequence->split(node,offset,&new_node))
    edit_history->Push(SPLIT,new_node->start,NULL);
  ins_node = new PieceTableNode(loci_add, n, loci_add);
  JoinPieceTableNode(new_node->pPrev, ins_node, new_node);
  edit_history->Push(INSERT,ins_node->start,ins_node);

  reference->search(new_node->start,&node,&offset);
  reference->split(node,offset,&new_node);
  ins_node = new PieceTableNode(loci_add, n, loci_add);
  JoinPieceTableNode(new_node->pPrev, ins_node, new_node);

  added_length += n;
}

void TextEditor::insert_break_in_history(){
  edit_history->Push(SEPARATOR,0,NULL);
}

void TextEditor::remove_n(int position, int n, bool bText, char *removed_text){
  int offset, off_temp;
  PieceTableNode *node, *from, *to, *current, *temp;

  sequence->search_pos(position,&node,&offset);

  if(sequence->split(node,offset,&from))
    edit_history->Push(SPLIT,from->start,NULL);
  sequence->move(from,0,n,&temp,&off_temp);
  if(sequence->split(temp,off_temp,&to))
    edit_history->Push(SPLIT,to->start,NULL);

  LinkPieceTableNode(from->pPrev,to);
  for(current=from;current!=to;){
    if(bText)
      get_text_from_loci(removed_text, current->start, current->length);
    edit_history->Push(DELETE,current->pNext->start,current);
    temp = current;
    current=current->pNext;
    delete temp;
  }
}

void TextEditor::replace_n(int position, int n, char *add, bool bText, char *replaced_text){
  strncat(buffer,add,n);
  replace_blank_n(position, n, bText, replaced_text);
}

void TextEditor::replace_blank_n(int position, int n, bool bText, char *replaced_text){
  int offset, off_temp, new_start, cum=0;
  PieceTableNode *node, *from, *to, *current, *temp;

  sequence->search_pos(position,&node,&offset);

  if(sequence->split(node,offset,&from))
    edit_history->Push(SPLIT,from->start,NULL);
  sequence->move(from,0,n,&temp,&off_temp);
  if(sequence->split(temp,off_temp,&to))
    edit_history->Push(SPLIT,to->start,NULL);

  for(current=from;current!=to;current=current->pNext){
    if(bText)
      get_text_from_loci(replaced_text, current->start, current->length);
    new_start = original_length+added_length+cum;
    edit_history->Push(REPLACE,new_start,current);
    current->start = new_start;
    cum += current->length;
  }
  added_length += n;
}

void TextEditor::undo(){
  int loci, offset;
  PieceTableNode *prev, *temp, *node;
  EditHistoryNode *edit;
  
  edit = edit_history->head;
  sequence->search(edit->loci,&node,&offset);
  prev = node->pPrev;
  switch(edit->type){
    case SPLIT:
      prev->length += node->length;
      temp = node->pNext;
      LinkPieceTableNode(prev,temp);
      delete node;
      break;
    case INSERT:
      temp = node->pNext;
      LinkPieceTableNode(prev,temp);
      delete node;
      break;
    case DELETE:
      temp = new PieceTableNode(edit->start,edit->length,edit->replaced_start);
      JoinPieceTableNode(prev,temp,node);
      break;
    case REPLACE:
      node->start = edit->start;
      node->replaced_start = edit->replaced_start;
      break;
  }
  edit_history->Pop();
}

void TextEditor::get_text_from_loci(char *seq, int loci, int n){
  if(loci < original_length)
    strncat(seq,original+loci,n);
  else
    strncat(seq,buffer+loci-original_length,n);
}

void TextEditor::get_text(PieceTable *piecetable, char *seq){
  PieceTableNode *current;
  for(current = piecetable->head->pNext;current!=piecetable->tail;current=current->pNext)
    get_text_from_loci(seq, current->start, current->length);
}

void TextEditor::get_original_empty(char *seq){
  PieceTableNode *current;
  int k, start=0;
  seq[0]='\0';
  for(current = reference->head->pNext;current!=NULL;current=current->pNext){
    if(current->start<original_length)
      get_text_from_loci(seq, current->start, current->length);
    else{
      for(k=0;k<current->length;k++)
        seq[start+k]='-';
      seq[start+current->length]='\0';
    }
    start += current->length;
  }
}

// loci_seq[i] stores the loci id at position i
// pos_seq[i] stores the position at loci i
void TextEditor::get_loci_seq(int *loci_seq, int *pos_seq){
  int j, cum_len=0;
  PieceTableNode *current;
  for(current = reference->head->pNext;current!=NULL;current=current->pNext){
    for(j=0;j<current->length;j++){
      loci_seq[cum_len] = current->start+j;
      pos_seq[current->start+j] = cum_len;
      cum_len ++;
    }
  }
}

void TextEditor::testprint(char *output){
  sequence->print();
  output[0] = '\0';
  get_text(sequence,output);
  printf("\nString in the text editor:\n%s",output);
  output[0] = '\0';
  get_text(reference,output);
  printf("\nString in the text reference:\n%s",output);
  printf("\nString in the buffer:\n%s",buffer);
  std::cin.get();
}

#endif
