#ifndef LOCALALIGN
#define LOCALALIGN

#include <iostream>                                                                                                                                          
#include <stdio.h>
#include <math.h>

#define NUMBER 4  // Number of possible letters. 4 for DNA sequence. 20 for amino acid sequence.
#define W 8       // Length of a tuple

#define LOCAL  0
#define SPARSE 1

/*******************************************************
    Consecutive W letters in Sequence A is called a tuple.

    A tuple is treated as a 4-ary number
    with digits A = 0, T = 1, C = 2, G = 3.
    This 4-ary number is then translated to a decimal number.
    
    A hash table (QueryTable) will be used to record
    all tuples appearing in Sequence A.
 *******************************************************/

struct QueryTableNode{
	unsigned long A_start;
	struct QueryTableNode *m_pNext;
};



/*******************************************************
    The results of local alignment are represented
    as a chain of fragments arranged in the
    ascending order of B_start, then
    descending order of A_start.
    
    Example:
    
            0000000000111111111122222222
            0123456789012345678901234567
    Seq A = TCGCATCTGCAGAAGCATCTGCAGAA
    Seq B = GCATCTGCAGAATCTCGCATCTGCAGAA
    
    The chain of fragments is
    
    (14,0,12)->(2,0,12)->(0,14,14)->(14,16,12)

 *******************************************************/

class LocalFragmentNode{
  public:
           
  unsigned long A_start;
  unsigned long B_start;
  unsigned int length;
  LocalFragmentNode *m_pNext;

  // The followings are not used in local alignment,
  // but they are used in sparse dynamic programming.
  unsigned int score;
  LocalFragmentNode *chain_to, *visa, *visl;
	
  LocalFragmentNode(unsigned long a, unsigned long b, unsigned int c, LocalFragmentNode *d);
};

LocalFragmentNode::LocalFragmentNode(unsigned long a, unsigned long b, unsigned int c, LocalFragmentNode *d){
  A_start = a;
  B_start = b;
  length = c;
  m_pNext = d;
}




class LocalFragmentChain{
  public:
  
  LocalFragmentNode *head, *last, *sparse_head;
  
  LocalFragmentChain();
  ~LocalFragmentChain();
  void FreeLocalFragment();
  void insert(LocalFragmentNode *x);
  void print(int blocal);
};

LocalFragmentChain::LocalFragmentChain(){
  head = NULL;
  last = NULL;
}

LocalFragmentChain::~LocalFragmentChain(){
  FreeLocalFragment();
}

void LocalFragmentChain::FreeLocalFragment(){
	LocalFragmentNode *temp;
	for(;head!=NULL;){
		temp=head;
        head=head->m_pNext;
		delete temp;
    }
    head = NULL;
    last = NULL;
}

void LocalFragmentChain::insert(LocalFragmentNode *x){
  if(head==NULL)
    head=x;
  else
    last->m_pNext=x;
  last=x;
}

void LocalFragmentChain::print(int blocal){
  LocalFragmentNode *current;

  if(blocal==LOCAL){
    for(current = head; current!=NULL; current = current->m_pNext)
      std::cout<<" A: from "<<current-> A_start<<" to "
        <<current-> A_start+current->length-1
        <<" B: from "<<current-> B_start<<" to "
        <<current-> B_start+current->length-1
        <<" length: "<<current->length<<std::endl;
  }else{
    for(current = sparse_head; current!=NULL; current = current->chain_to)
      std::cout<<" A: from "<<current-> A_start<<" to "
        <<current-> A_start+current->length-1
        <<" B: from "<<current-> B_start<<" to "
        <<current-> B_start+current->length-1
        <<" length: "<<current->length<<std::endl;
  }
}





unsigned long tablelength=(unsigned long)pow(NUMBER,W);
unsigned long coefficient=(unsigned long)pow(NUMBER,W-1);
int threshold = 20;      // Ignore local alignments with length < threshold


class LocalAlignment{
  public:
  
  LocalAlignment();
  ~LocalAlignment();

  void Reset();
  void LocalAlign(char *input1, char *input2, int n1, int n2, LocalFragmentChain *HitTable);

  private:
  
  char *SequenceA, *SequenceB;             // Sequence A = Query sequence
  int  LengthSequenceA, LengthSequenceB;
  LocalFragmentChain *HitTable;

  struct QueryTableNode **pTable;

  void AllocateQuery();
  void FreeQueryTable();

  unsigned int LetterToValue(char letter) ;
  void CounterValue(unsigned int x_old,unsigned long *value,unsigned int x_new);
  void AddQueryToTable();

  int  ExtendHit(LocalFragmentNode *new_hit);
  int  TestHit(LocalFragmentNode *L_temp, LocalFragmentNode *temp);
  int  TestHitTable(LocalFragmentNode *new_hit);
};


/*******************************************************
     Local alignment
     Constructor and destructor
 *******************************************************/


LocalAlignment::LocalAlignment(){
  AllocateQuery();
}

LocalAlignment::~LocalAlignment(){
  FreeQueryTable();
}

void LocalAlignment::AllocateQuery(){
  unsigned long index;

  pTable=(struct QueryTableNode **)malloc(tablelength*sizeof(struct QueryTableNode *));
  for(index=0;index<tablelength;index++)
    pTable[index]=NULL;
}

void LocalAlignment::FreeQueryTable(){
	struct QueryTableNode *temp_table;
	unsigned long index;
	for(index=0;index<tablelength;index++)
     for(;pTable[index]!=NULL;){
		temp_table=pTable[index];
        pTable[index]=pTable[index]->m_pNext;
		free(temp_table);
     }
	free(pTable);
}

void LocalAlignment::Reset(){
  FreeQueryTable();
  AllocateQuery();
}


/*******************************************************
     Local alignment
     Reading a sequence into the query table
 *******************************************************/

unsigned int LocalAlignment::LetterToValue(char letter){
	unsigned int value; 
	switch(letter)
	{case 'A':value=0;break;
	 case 'T':value=1;break;
	 case 'C':value=2;break;
     case 'G':value=3;break;
     default :value=0;break;
	}
	return(value);
}

void LocalAlignment::CounterValue(unsigned int x_old,unsigned long *value,unsigned int x_new){
  (*value)-=x_old*coefficient;
  (*value)*=NUMBER;
  (*value)+=x_new;
  return;
}

void LocalAlignment::AddQueryToTable(){
  struct QueryTableNode *new_QueryWord;
  unsigned long i,value;
  char temp_old,temp_new;

  value=0;
  i=0;

  for(i=0;i<W;i++)
  {   
	  temp_new=SequenceA[i];
	  value=value*NUMBER+LetterToValue(temp_new);
  }
  pTable[value]=(struct QueryTableNode *)malloc(sizeof(struct QueryTableNode));
  pTable[value]->A_start=0;
  pTable[value]->m_pNext=NULL;
  
  for(i=1;i<=LengthSequenceA-W;i++)
  {
	  temp_old=SequenceA[i-1];
	  temp_new=SequenceA[i+W-1];
      CounterValue(LetterToValue(temp_old),&value,LetterToValue(temp_new));
	  new_QueryWord=(struct QueryTableNode *)malloc(sizeof(struct QueryTableNode));
	  new_QueryWord->m_pNext=pTable[value];
      pTable[value]=new_QueryWord;
	  pTable[value]->A_start=i;
  }
  return;
}

 
/*******************************************************
     Local alignment
     Compare Sequence B with the query table
 *******************************************************/



// Extend the matched tuple to make a longer fragment of local alignment
// Keep the fragment if the length after extention is bigger than the threshold
int LocalAlignment::ExtendHit(LocalFragmentNode *new_hit){
	unsigned long A_index, B_index;

	A_index = new_hit->A_start+new_hit->length-1;
	B_index = new_hit->B_start+new_hit->length-1;
    for(;(A_index<LengthSequenceA-1)&&(B_index<LengthSequenceB-1);)// right extend
	{
		if(SequenceA[A_index+1]==SequenceB[B_index+1]){
            A_index++;
            B_index++;
			new_hit->length++;
		}else
			break;
	}
	if((new_hit->length)<threshold)
		return(0);
	else
	   return(1);
}

// Test if the tuple temp is part of previously extended fragment L_temp in the hit table.
// If so, don't extend.
int LocalAlignment::TestHit(LocalFragmentNode *L_temp, LocalFragmentNode *temp){
	if((temp->A_start-temp->B_start)==(L_temp->A_start-L_temp->B_start)){
      if((temp->A_start)>=(L_temp->A_start)){
		 if((temp->A_start)<(L_temp->A_start+L_temp->length))
            return(0); //don't extend because it is part of the previously extended fragment.
	  }
	}
	return(1); //it is not part of the previously extended fragment.
}

// Test if the tuple temp is part of any previously extended fragment in the hit table.
// If so, don't extend.
int LocalAlignment::TestHitTable(LocalFragmentNode *new_hit)
{
	LocalFragmentNode *temp;
	if(HitTable->head==NULL)
		return(1);
	temp=HitTable->head;
	do{
		if(TestHit(temp,new_hit)==0)
			return(0);
        temp=temp->m_pNext;
	}while(temp!=NULL);
	return(1);
}

void LocalAlignment::LocalAlign(char *input1, char *input2, int n1, int n2, LocalFragmentChain *result){
  struct QueryTableNode *p_QueryTableValue;
  LocalFragmentNode *new_hit;
  unsigned long i,index_SequenceB,value;
  char temp_old,temp_new;
  bool bInserted;
  
  SequenceA = input1;
  SequenceB = input2;
  LengthSequenceA = n1;
  LengthSequenceB = n2;
  HitTable = result;
  
  AddQueryToTable();


  value=0;
  for(index_SequenceB=0;index_SequenceB<=LengthSequenceB-W;index_SequenceB++)
  {
      if(index_SequenceB==0){
          for(i=0;i<W;i++){   
	        temp_new=SequenceB[i];
	        value=value*NUMBER+LetterToValue(temp_new);
          }
      }else{
        temp_old=SequenceB[index_SequenceB-1];
	    temp_new=SequenceB[index_SequenceB+W-1];
        CounterValue(LetterToValue(temp_old),&value,LetterToValue(temp_new));
      }
      
      if(pTable[value]!=NULL)
	  {   
	   p_QueryTableValue=pTable[value];
       do{
	    new_hit = new LocalFragmentNode(p_QueryTableValue->A_start,index_SequenceB,W,NULL);
        bInserted=false;
		if(TestHitTable(new_hit)==1){
		 if(ExtendHit(new_hit)==1){
           HitTable->insert(new_hit);
           bInserted=true;
		 }
        }
	    if(!bInserted)
          delete(new_hit);

		 p_QueryTableValue=p_QueryTableValue->m_pNext;
	   }while(p_QueryTableValue!=NULL);
	  }
  }
}


#endif
