#include <iostream>                                                                                                                                          
#include <malloc.h>

#include "maxmin.cpp"

#ifndef AVLTREE
#define AVLTREE

#define EQ 0
#define GT 1
#define LT 2
#define LE 3
#define GE 4

#define LEFT 0
#define RIGHT 1

/*******************************************************
             Class: Node of an AVL tree
 *******************************************************/

template <class T>
class AVLnode{
  public:

  T     key;      // Variables for sorting
  void  *data;
  
  AVLnode<T> *parent, *child1, *child2;

  // BF = balance factor of a node
  // height = height of a node in the AVL tree
  int height, BF;
  
  public:
  
  AVLnode();           // Create an isolated node without specifying the value
  AVLnode(T val, void *x);      // Create an isolated node with initial value

  void SetData(T val, void *x); // Change the value of a node without changing the tree topology

  void print();        // Print the value and height and BF
  void print_value();  // Print the value only
};

/*******************************************************
             Constructors
 *******************************************************/

template <class T> AVLnode<T>::AVLnode(){
  height=1;
  BF=0;
  parent=NULL;
  child1=NULL;
  child2=NULL;
}

template <class T> AVLnode<T>::AVLnode(T val, void *x){
  key = val;
  data = x;
  height=1;
  BF=0;
  parent=NULL;
  child1=NULL;
  child2=NULL;
}

template <class T> void AVLnode<T>::SetData(T val, void *x){
  key = val;
  data = x;  
}

/*******************************************************
             Printing the data
 *******************************************************/

template <class T> void AVLnode<T>::print(){
  std::cout<<key<<"  "<<BF<<"  "<<height;
}

template <class T>
void AVLnode<T>::print_value(){
  std::cout<<key;
}








/*******************************************************
             Class: AVL tree
   Here the root of the tree is pointed to by *par_root
 *******************************************************/

template <class T>
class AVLtree{
  public:

  AVLnode<T> par_root;          // A null node with the root as the left child

  private:
  
  bool *indentation;             // Used in outputting the tree in a text
  
  public:
  
  AVLtree();
  ~AVLtree();

  void insert(T val, void *x);                             // Create a node with value val and insert the new node into the tree. Here, x stores all attributes other than the key
  template <class U> void remove(AVLnode<U> *node);        // Delete the node
  template <class U> AVLnode<U>* search(U val, int mode);  // Search for the node with value val.
  /* mode: EQ -- search for node = val
           LT -- search for largest node < val
           LE -- search for largest node <= val
           GT -- search for largest node > val
           GE -- search for largest node >= val */

  void print_sorted_list();   // Print the nodes in ascending order of val
  void print();               // Print the tree
  
  private:

  // Update height and BF. Return true if changed. It's called after tree operations.
  template <class U> bool updated(AVLnode<U> *current);
  // dir: LEFT (RIGHT) -- Set "child" as the left (right) child of "parent"
  template <class U> void join(AVLnode<U> *child, AVLnode<U> *parent, int dir);
  // Rotate the tree to keep |BF| not greater than 1
  template <class U> void rotate(AVLnode<U> *grandparent, int dir);
  template <class U> void parse(AVLnode<U> *node);
  template <class U> void free_tree_from_node(AVLnode<U> *node);
  template <class U> void print_tree_from_node(AVLnode<U> *node, int depth);
};

/*******************************************************
             Constructors and destructors
 *******************************************************/

template <class T>
AVLtree<T>::AVLtree(){
  indentation = (bool*)malloc((unsigned) 300*sizeof(bool));
}

template <class T>
AVLtree<T>::~AVLtree(){
  free_tree_from_node(par_root.child1);
  free(indentation);
}

template <class T> template <class U>
void AVLtree<T>::free_tree_from_node(AVLnode<U> *node){
  if(node==NULL)
    return;
  free_tree_from_node(node->child1);
  free_tree_from_node(node->child2);
  delete(node);
}

/*******************************************************
             Functions for doing rotation
 *******************************************************/

template <class T> template <class U>
bool AVLtree<T>::updated(AVLnode<U> *current){
  int h1, h2, temp;
  
  if(current->child1!=NULL)
    h1 = current->child1->height;
  else
    h1 = 0;
  if(current->child2!=NULL)
    h2 = current->child2->height;
  else
    h2 = 0;
  temp = current->height;
  current->height = 1+max(h1,h2);
  current->BF = h1-h2;
  if(temp==current->height)
    return false;
  return true;
}

template <class T> template <class U>
void AVLtree<T>::join(AVLnode<U> *child, AVLnode<U> *parent, int dir){
  if(parent!=NULL){
    if(dir==LEFT)
      parent->child1=child;
    else if(dir==RIGHT)
      parent->child2=child;
  }
  if(child!=NULL)
    child->parent=parent;
}

template <class T> template <class U>
void AVLtree<T>::rotate(AVLnode<U> *grandparent, int dir){
  AVLnode<U> *A, *B, *C, *b, *b1, *b2;
  
  if(dir==LEFT)
    A = grandparent->child1;
  else if(dir==RIGHT)
    A = grandparent->child2;

  if(A->BF>0){                   // Left is heavier
    if(A->child1->BF>=0){         // LL type.
/*
             A                                     B
            / \                                   / \
           /   \                                 /   \
          B     a           ==>                 c     A
         / \                                         / \
        /   \                                       /   \
       c     b                                     b     a
*/
      B = A->child1;
      b = B->child2;
      
      join(B,grandparent,dir);
      join(A,B,RIGHT);
      join(b,A,LEFT);
      updated(A);
      updated(B);

    }else{   // LR type
/*
                 A                                        B
                / \                                      / \
               /   \                                    /   \
              C     a              ==>                 C     A
             /  \                                     / \   / \
            /    \                                   /   |  |  \
           c     B                                  c   b2  b1  a
                / \
               /   \
              b2   b1
*/
      C  = A->child1;
      B  = C->child2;
      b1 = B->child2;
      b2 = B->child1;
      
      join(B,grandparent,dir);
      join(C,B,LEFT);
      join(A,B,RIGHT);
      join(b2,C,RIGHT);
      join(b1,A,LEFT);
      updated(A);
      updated(C);
      updated(B);
    }
  }else if(A->BF<0){             // Right is heavier
    if(A->child2->BF<=0){         // RR type.
/*
             A                                     B
            / \                                   / \
           /   \                                 /   \
          a     B           ==>                 A     c
               / \                             / \
              /   \                           /   \
             b     c                         a     b
*/
      B = A->child2;
      b = B->child1;
      
      join(B,grandparent,dir);
      join(A,B,LEFT);
      join(b,A,RIGHT);
      updated(A);
      updated(B);
    
    }else{   // RL type
/*
                 A                                        B
                / \                                      / \
               /   \                                    /   \
              a     C              ==>                 A     C
                   / \                                / \   / \
                  /   \                              /   |  |  \
                 B     c                            a   b1  b2  c
                / \
               /   \
              b1   b2
*/
      C  = A->child2;
      B  = C->child1;
      b1 = B->child1;
      b2 = B->child2;
      
      join(B,grandparent,dir);
      join(C,B,RIGHT);
      join(A,B,LEFT);
      join(b2,C,LEFT);
      join(b1,A,RIGHT);
      updated(A);
      updated(C);
      updated(B);
    }
  }
}

/*******************************************************
            Tree operations: insert, remove, search
 *******************************************************/

template <class T>
void AVLtree<T>::insert(T val, void *x){
  AVLnode<T> *new_node, *current;
  int h1, h2;         // Heights of left subtree and right subtree
  int temp;

  new_node = new AVLnode<T>(val, x);
  if((par_root.child1==NULL)&&(par_root.child2==NULL)){
    join(new_node,&par_root,LEFT);
    return;
  }
  
  // Search for the correct place to insert.
  for(current = par_root.child1;;){
    if(val >= current->key){
      if(current->child2==NULL){
        join(new_node,current,RIGHT);
        break;
      }else
        current = current->child2;
    }else{
      if(current->child1==NULL){
        join(new_node,current,LEFT);
        break;
      }else
        current = current->child1;
    }
  }
  
  // Update the heights and the balance factors.
  for(;current->parent!=NULL;current=current->parent){
    // Update the height of the subtrees.
    if(!updated(current))
      break;
    if(abs(current->BF)>1){
      // Do the rotation to keep the AVL tree balanced. Return the pointer of root if the root has changed.
      if(current==current->parent->child1)
        rotate(current->parent,LEFT);
      else if(current==current->parent->child2)
        rotate(current->parent,RIGHT);
      return;
    }
  }
}

template <class T> template <class U>
void AVLtree<T>::remove(AVLnode<U> *node){
  AVLnode<U> *current, *tmpNode;
  int h1, h2;         // Heights of left subtree and right subtree
  int temp;

  if(node==NULL)
    return;
    
  if((node->child1==NULL)&&(node->child2==NULL)){
    if(node==node->parent->child1)
      join((AVLnode<U>*)NULL,node->parent,LEFT);
    else if(node==node->parent->child2)
      join((AVLnode<U>*)NULL,node->parent,RIGHT);
    current = node->parent;
    delete(node);
  }
  else if((node->child1!=NULL)&&(node->child2==NULL)){
    // Replace the deleted node by its child
    if(node==node->parent->child1)
      join(node->child1,node->parent,LEFT);
    else if(node==node->parent->child2)
      join(node->child1,node->parent,RIGHT);
    current = node->parent;
    delete(node);
  }
  else if((node->child1==NULL)&&(node->child2!=NULL)){
    // Replace the deleted node by its child
    if(node==node->parent->child1)
      join(node->child2,node->parent,LEFT);
    else if(node==node->parent->child2)
      join(node->child2,node->parent,RIGHT);
    current = node->parent;
    delete(node);
  }
  else{
    // Search for the maximum node on the left subtree.
    for(current=node->child1; current->child2!=NULL; current=current->child2);
    // Replace the deleted node by the max node.
    node->SetData(current->key,current->data);
    // Delete the max node.
    if(current==current->parent->child1)
      join(current->child1,current->parent,LEFT);
    else if(current==current->parent->child2)
      join(current->child1,current->parent,RIGHT);
    tmpNode = current->parent;
    delete(current);
    current = tmpNode;
  }
  // Update the heights and the balance factors.
  for(;current!=&par_root;current=current->parent){
    // Update the height of the subtrees.
    updated(current);
    if(abs(current->BF)>1){
      // Do the rotation to keep the AVL tree balanced.
      if(current==current->parent->child1)
        rotate(current->parent,LEFT);
      else if(current==current->parent->child2)
        rotate(current->parent,RIGHT);
    }
  }
}

template <class T> template <class U>
AVLnode<U>* AVLtree<T>::search(U val, int mode){
  AVLnode<U> *current, *temp;
  
  current=par_root.child1;
  
  switch(mode){
  
  case EQ:  // Search for the node equal to val.
  
  while(current!=NULL){
    if(val > current->key)
      current = current->child2;
    else if(val==current->key)
      return current;
    else
      current = current->child1;
  }
  return NULL;
  
  case LT:  // Search for the largest node smaller than node.
  
  temp = NULL;
  while(current!=NULL){
    if(val > current->key){
       temp = current;
       current = current->child2;
    }else
       current = current->child1;
  }
  return temp;

  case LE:  // Search for the largest node smaller than or equal to node.
  
  temp = NULL;
  while(current!=NULL){
    if(val > current->key){
      temp = current;
      current = current->child2;
    }else if(val==current->key)
      return current;
    else
      current = current->child1;
  }
  return temp;

  case GT:  // Search for the smallest node greater than node.
  
  temp = NULL;
  while(current!=NULL){
    if(val < current->key){
       temp = current;
       current = current->child1;
    }else
       current = current->child2;
  }
  return temp;

  case GE:  // Search for the smallest node greater than or equal to node.
  
  temp = NULL;
  while(current!=NULL){
    if(val < current->key){
      temp = current;
      current = current->child1;
    }else if(val==current->key)
      return current;
    else
      current = current->child2;
  }
  return temp;

  }
}

/*******************************************************
         Output the result of tree sort
 *******************************************************/

// Recursive function for reading the tree
template <class T> template <class U>
void AVLtree<T>::parse(AVLnode<U> *root){
  if(root==NULL)
    return;
  parse(root->child1);
  root->print_value();
  std::cout<<" ";
  parse(root->child2);
}

template <class T>
void AVLtree<T>::print_sorted_list(){
  parse(par_root.child1);
}

/*******************************************************
         Print the tree
 *******************************************************/

// Recursive function for reading the tree
template <class T> template <class U>
void AVLtree<T>::print_tree_from_node(AVLnode<U> *root, int depth){
  int j;
  
  indentation[depth+1]=false;

  if(root->child2!=NULL){
    print_tree_from_node(root->child2,depth+1);
    indentation[depth+1]=true;
  }
  if(root->child1!=NULL){
    print_tree_from_node(root->child1,depth+1);
    indentation[depth+1]=true;
  }
 
  for(j=0;j<depth;j++){
    if(indentation[j])
      std::cout<<"|   ";
    else
      std::cout<<"    ";
  }
  std::cout<<" --- (";
  root->print();
  std::cout<<")\n";
  for(j=0;j<depth;j++){
    if(indentation[j])
      std::cout<<"|   ";
    else
      std::cout<<"    ";
  }
  std::cout<<"|"<<std::endl;
}

template <class T>
void AVLtree<T>::print(){
  indentation[0]=false;
  print_tree_from_node(par_root.child1,0);
}

#endif
