#ifndef SPARSEALIGN
#define SPARSEALIGN

#include <iostream.h>                                                                                                                                          
#include <malloc.h>

#include "AVLTree.cpp"
#include "localalign.cpp"

#define VISA  LT
#define VISL  GT


/*******************************************************
      Class: Twoint. To be used in
        Sparse dynamic programming
 *******************************************************/

class Twoints{
  public:
  
  int x,y;
  bool bStart;
  
  public:
  
  Twoints(){};
  Twoints(int a, int a2, bool a3);
  bool operator>(const Twoints &other);
  bool operator<(const Twoints &other);
  bool operator>=(const Twoints &other);
  bool operator<=(const Twoints &other);
  bool operator==(const Twoints &other);
};

Twoints::Twoints(int a, int a2, bool a3){
  bStart=a3;
  if(bStart){ x=2*a; y=2*a2; }
  else { x=2*a+1; y=2*a2+1; }
}

bool Twoints::operator>(const Twoints &other){
  if(x > other.x)
    return true;
  if(x == other.x)
    if(y < other.y)
      return true;
  return false;
}

bool Twoints::operator<(const Twoints &other){
  if(x < other.x)
    return true;
  if(x == other.x)
    if(y > other.y)
      return true;
  return false;
}

bool Twoints::operator==(const Twoints &other){
  return (x == other.x)&&(y == other.y);
}

bool Twoints::operator>=(const Twoints &other){
  return !(*this < other);
}
bool Twoints::operator<=(const Twoints &other){
  return !(*this > other);
}




/*******************************************************
    The program codes implement the sparse dynamic
    programming algorithm in Baker and Giancarlo, 1999.
    Purpose: To chain the local alignments together.
 *******************************************************/


class SparseDynamicProgramming{
  public:
  
  SparseDynamicProgramming();
  void sparse_DP(LocalFragmentChain *result);

  private:

  AVLtree<Twoints> *X_tree;
  AVLtree<int> *L_tree;

  void find_vis_subtree(AVLnode<Twoints> *xtree, int bleft);
  void find_vis(int bleft);
  void scan_X_subtree(AVLnode<Twoints> *xtree);
  void scan_X_tree();
};



SparseDynamicProgramming::SparseDynamicProgramming(){
}


// For each fragment (can be expressed as a square in the n1 x n2 matrix),
// find the closest fragment that is visable on the left
// and the closest fragment that is visable above.

void SparseDynamicProgramming::find_vis(int bleft){
  L_tree = new AVLtree<int>;
  if(X_tree->par_root.child1!=NULL)
    find_vis_subtree(X_tree->par_root.child1,bleft);
  delete L_tree;
}

void SparseDynamicProgramming::find_vis_subtree(AVLnode<Twoints> *xtree, int bleft){
  int temp;
  AVLnode<int> *target;
  LocalFragmentNode *current;
  bool bNeedInsert=false;

  if(xtree==NULL)
    return;

  find_vis_subtree(xtree->child1,bleft);  // Do recursion.
  
  current = (LocalFragmentNode*)(xtree->data);
  if(xtree->key.bStart){  // A begin event.
    temp = current->A_start-current->B_start;
    target = L_tree->search(temp,bleft);
    L_tree->insert(temp,current);
    
    if(target!=NULL){
      if(bleft==VISA)
        current->visa = (LocalFragmentNode*)(target->data);
      else
        current->visl = (LocalFragmentNode*)(target->data);
    }else{
      if(bleft==VISA)
        current->visa = NULL;
      else
        current->visl = NULL;
    }
  }else{  // An end event.
    temp = current->A_start-current->B_start;
    target = L_tree->search(temp,GE);
    L_tree->remove(target);
  }
  
  find_vis_subtree(xtree->child2,bleft);   // Do recursion
}



void SparseDynamicProgramming::scan_X_subtree(AVLnode<Twoints> *xtree){
  int temp;
  AVLnode<int> *target;
  LocalFragmentNode *current, *preceding, *visa, *visl;
  int numcomp[3],which;
  bool bNeedInsert=false;

  if(xtree==NULL)
    return;
  scan_X_subtree(xtree->child1);  // Do recursion.
  
  current = (LocalFragmentNode*)(xtree->data);

  if(xtree->key.bStart){  // A begin event.
    temp = current->A_start;    // Find largest l in L_tree that is smaller than current h
    target = L_tree->search(temp,LT);
    if(target!=NULL)
      preceding = (LocalFragmentNode*)(target->data);
    else
      preceding = NULL;
    visa = current->visa;
    visl = current->visl;

    if(preceding!=NULL)
      numcomp[0] = preceding->score + current->length;
    else
      numcomp[0] = current->length;

    if(visa!=NULL)
      numcomp[1] = visa->score + current->length - ( visa->B_start+visa->length-1 - current->B_start);
    else
      numcomp[1] = current->length;
    
    if(visl!=NULL)
      numcomp[2] = visl->score + current->length - ( visl->A_start+visl->length-1 - current->A_start);
    else
      numcomp[2] = current->length;
    
    which = which_max(3,numcomp);
    current->chain_to = NULL;
    switch(which){
      case 0: if(preceding!=NULL) current->chain_to = preceding; break;
      case 1: if(visa!=NULL) current->chain_to = visa; break;
      case 2: if(visl!=NULL) current->chain_to = visl; break;
    }
    current->score = numcomp[which];
    
  }else{  // An end event.
    temp = current->A_start+current->length-1;    // Find largest l in L_tree that is smaller than current l
    target = L_tree->search(temp,LT);
    if(target==NULL)
      bNeedInsert=true;
    else if(current->score > ((LocalFragmentNode*)(target->data))->score)
      bNeedInsert=true;

    if(bNeedInsert){
      target = L_tree->search(temp,GE);
      while(target!=NULL){
        if(((LocalFragmentNode*)(target->data))->score > current->score)
          break;
        L_tree->remove(target);
        target = L_tree->search(temp,GE);
      }
      L_tree->insert(temp,current);
    }
  }
  scan_X_subtree(xtree->child2);   // Do recursion
}

void SparseDynamicProgramming::scan_X_tree(){
  scan_X_subtree(X_tree->par_root.child1);
}

void SparseDynamicProgramming::sparse_DP(LocalFragmentChain *result){
  LocalFragmentNode *current;
  AVLnode<int> *current2;

  X_tree = new AVLtree<Twoints>;

  // Sort the start events and end events in ascending order of x-coordinate values
  current = result->head;
  while(current!=NULL){
    X_tree->insert(Twoints(current->A_start,current->B_start,true),current);
    X_tree->insert(Twoints(current->A_start+current->length-1,current->B_start+current->length-1,false),current);
    current=current->m_pNext;
  }
  find_vis(VISL);
  delete X_tree;

  X_tree = new AVLtree<Twoints>;
  current = result->head;
  while(current!=NULL){
    X_tree->insert(Twoints(current->B_start,current->A_start,true),current);
    X_tree->insert(Twoints(current->B_start+current->length-1,current->A_start+current->length-1,false),current);
    current=current->m_pNext;
  }
  find_vis(VISA);


  L_tree = new AVLtree<int>;
  scan_X_tree();
  // Search for the node with smallest h value.
  if(L_tree->par_root.child1!=NULL){
    for(current2=L_tree->par_root.child1;current2->child2!=NULL;current2=current2->child2);
      current = (LocalFragmentNode*)(current2->data);
  }else
    current = NULL;
  delete L_tree;
  delete X_tree;

  result->sparse_head = current;
}



#endif
