#ifndef SIMTREE
#define SIMTREE

#include "phylotree.cpp"
#include "piecetable.cpp"


char letter[4] = {'A','T','C','G'};

// Substitution matrix for A G C T
double sub_rate[4][4]=
{{0,    0.9,  0.05, 0.05},
 {0.9,  0,    0.05, 0.05},
 {0.05, 0.05, 0,    0.9},
 {0.05, 0.05, 0.9,  0}};
// Transition probability matrix for A G C T. Take DNA sequence as Markov chain time series
double transition[4][4]=
{{0.4,  0.2,  0.2,  0.2},
 {0.2,  0.4,  0.2,  0.2},
 {0.2,  0.2,  0.4,  0.2},
 {0.2,  0.2,  0.2,  0.4}};
// Initial distribution of A G C T
double letter_prob[4]={0.25,0.25,0.25,0.25};
double type_prob[3]={0.8,0.1,0.1}; // Probability that the mutation is POI, INS, and DEL.
double m_gap=3;                 // Mean size of the indel gap.
double m_mutations=20;          // Mean number of mutations from the root.


/*******************************************************
             Random number generators
 *******************************************************/

double runif(){
  return rand()/32767.;
}

int rPoisson(double m){
  double p,cp,u;
  int k;
  
  k=0;
  p=cp=exp(-m);
  u=runif();
  while(u>cp){
    k++;
    p=(p*m)/k;
    cp+=p;
  }
  return k;
}

int rsample(int n, double *prob){
  int k;
  double u, cum_prob=0.;
  
  u=runif();
  for(k=0;k<n;k++){
    cum_prob += prob[k];
    if(u < cum_prob)
      return k;
  }
  return n-1;
}

unsigned int LetterToValue(char letter){
	unsigned int value; 
	switch(letter)
	{case 'A':value=0;break;
	 case 'T':value=1;break;
	 case 'C':value=2;break;
     case 'G':value=3;break;
     default :value=0;break;
	}
	return(value);
}



class tree_generator{
  public:

  tree_generator();
  ~tree_generator();

  void sim_tree(familytree *mytree, int n, char *input);
  void sim_tree_from_topology(familytree *mytree, char *input);
  
  private:
          
  TextEditor *editor;
  char *before_change, *after_change;

  void sim_num_mutations_from_nodes(phylonode *mynode);
  void sim_markers_from_nodes(phylonode *mynode,int parent_length);
  void sim_markers(int m, phylonode *mynode);
  void sim_substitution();
  void sim_insert_random_codes(int n);
  void write_markers(phylonode *mynode);
};

/*******************************************************
        Constructor and destructor
 *******************************************************/

tree_generator::tree_generator(){
  before_change = (char*)malloc((unsigned) 100*sizeof(char));
  after_change = (char*)malloc((unsigned) 100*sizeof(char));
  editor = new TextEditor();
}

tree_generator::~tree_generator(){
  free(before_change);
  free(after_change);
  delete editor;
}

  
/*******************************************************
             Simulations
 *******************************************************/

void tree_generator::sim_substitution(){
  int i,n, value, new_value;
  
  n = strlen(before_change);
  after_change[n] = '\0';
  for(i=0;i<n;i++){
    value = LetterToValue(before_change[i]);
    new_value = rsample(4,sub_rate[value]);
    after_change[i] = letter[new_value];
  }
}

void tree_generator::sim_insert_random_codes(int n){
  int k,value;
  
  value = rsample(4,letter_prob);
  after_change[0] = letter[value];
  for(k=1;k<n;k++){
    value = rsample(4,transition[value]);
    after_change[k] = letter[value];
  }
  after_change[n]='\0';
}

void tree_generator::write_markers(phylonode *mynode){
  int i;
  EditHistoryNode *hist;
  
  for(hist=editor->edit_history->head; hist->type!=SEPARATOR; hist=hist->pNext){
    switch(hist->type){
      case REPLACE:
        before_change[0] = after_change[0] = '\0';
        editor->get_text_from_loci(before_change,hist->start,hist->length);
        editor->get_text_from_loci(after_change,hist->loci,hist->length);
        for(i=hist->length-1;i>=0;i--)
          mynode->markers->m_pNext = new marker(hist->replaced_start+i,before_change[i],after_change[i],mynode->markers->m_pNext);
        break;
      case INSERT:
        after_change[0] = '\0';
        editor->get_text_from_loci(after_change,hist->loci,hist->length);
        for(i=hist->length-1;i>=0;i--)
          mynode->markers->m_pNext = new marker(hist->start+i,'-',after_change[i],mynode->markers->m_pNext);
        break;
      case DELETE:
        before_change[0] = '\0';
        editor->get_text_from_loci(before_change,hist->start,hist->length);
        for(i=hist->length-1;i>=0;i--)
          mynode->markers->m_pNext = new marker(hist->replaced_start+i,before_change[i],'-',mynode->markers->m_pNext);
        break;
    }
    editor->undo();
  }
  editor->undo();  // Undo the separator
}

void tree_generator::sim_markers_from_nodes(phylonode *mynode, int parent_length){
  int m,indel_size,new_length;
  int j,pos,type;
  marker *new_marker;

  new_length = parent_length;
  if(mynode->parent!=NULL){
    m = mynode->n_markers;
    editor->insert_break_in_history();
    for(j=0;j<m;j++){
      type = rsample(3,type_prob);
      switch(type){
        case 0:  // Point mutation event
          pos = rand()%new_length;
          before_change[0] = '\0';
          editor->replace_blank_n(pos,1,true,before_change);
          sim_substitution();
          strncat(editor->buffer,after_change,1);
          break;
        case 1:  // Insertion event
          pos = rand()%new_length;
          indel_size = min(rPoisson(m_gap-1)+1,99);
          sim_insert_random_codes(indel_size);
          editor->insert_n(pos,indel_size,after_change);
          new_length += indel_size;
          break;
        case 2:  // Deletion event
          pos = rand()%new_length;
          indel_size = min( min(rPoisson(m_gap-1)+1,99), new_length-pos);
          editor->remove_n(pos,indel_size,false,NULL);
          new_length -= indel_size;
          break;
      }
    }
  }

  if(mynode->child1!=NULL)
    sim_markers_from_nodes(mynode->child1,new_length);
  if(mynode->child2!=NULL)
    sim_markers_from_nodes(mynode->child2,new_length);

  if(mynode->parent!=NULL)
    write_markers(mynode);
}

void tree_generator::sim_num_mutations_from_nodes(phylonode *mynode){
  int m,indel_size;
  int j,pos,type;
  marker *new_marker;
  
  if(mynode->parent!=NULL)
    mynode->n_markers = rPoisson( m_mutations*fabs(mynode->parent->coalescent_time-mynode->coalescent_time)/2. );

  if(mynode->child1!=NULL)
    sim_num_mutations_from_nodes(mynode->child1);
  if(mynode->child2!=NULL)
    sim_num_mutations_from_nodes(mynode->child2);
}

void tree_generator::sim_tree(familytree *mytree, int n, char *input){
  int i,len,merge1,merge2;
  double T=0;
  phylonode *new_node;
  phylonode *temp,*left_branch,*right_branch;
  phylonode **coalescent;
  
  mytree->nseq = n;
  len = mytree->LengthSeqComb;
  coalescent =(phylonode**)malloc((unsigned) MAXK*sizeof(phylonode*));
  
  // Simulate the tree structure
  for(i=0;i<n;i++){
    mytree->table[i] = new phylonode(i);
    coalescent[i] = mytree->table[i];
    coalescent[i]->coalescent_time = 0;
  }
  for(i=n;i>1;i--){
    merge1=rand()%i;
    temp = coalescent[i-1];
    coalescent[i-1] = coalescent[merge1];
    coalescent[merge1] = temp;
    merge2=rand()%(i-1);
    new_node = new phylonode(-1);
    mytree->join_tree(coalescent[i-1],new_node,LEFT);
    mytree->join_tree(coalescent[merge2],new_node,RIGHT);
    T += 2.0/i/(i-1);
    new_node->coalescent_time = T;
    coalescent[merge2]=new_node;
  }
  temp = new phylonode(-1);
  temp->coalescent_time = T;
  mytree->join_tree(coalescent[0],temp,LEFT);
  mytree->par_root.child1 = temp;
  left_branch = coalescent[0]->child1;
  right_branch = coalescent[0]->child2;

  // Simulate the number of mutations
  sim_num_mutations_from_nodes(temp);

  // Choose any leaf as the root.
  mytree->SetRootTopologyOnly(0);
  
  // Merge the edges from the root to create an unrooted tree.
  delete temp;
  if(coalescent[0]->parent==left_branch){
    right_branch->n_markers += coalescent[0]->n_markers;
    if(coalescent[0]==left_branch->child1)
      mytree->join_tree(right_branch,left_branch,LEFT);
    else
      mytree->join_tree(right_branch,left_branch,RIGHT);
  }else{
    left_branch->n_markers += coalescent[0]->n_markers;
    if(coalescent[0]==right_branch->child1)
      mytree->join_tree(left_branch,right_branch,LEFT);
    else
      mytree->join_tree(left_branch,right_branch,RIGHT);
  }
  delete coalescent[0];
  free(coalescent);

  // Simulate the mutations
  sim_tree_from_topology(mytree,input);
}

  
void tree_generator::sim_tree_from_topology(familytree *mytree, char *input){
  char *tmpc;
  int len;
    
  // Generate mutations on a given tree
  len = strlen(input);
  editor->SetOriginal(input);
  sim_markers_from_nodes(mytree->par_root.child1,len);

  mytree->SeqComb[0] = '\0';
  editor->get_text(editor->reference,mytree->SeqComb);
  editor->get_original_empty(mytree->SeqRoot);
  editor->get_loci_seq(mytree->loci_id,mytree->pos_id);
  mytree->LengthSeqComb = strlen(mytree->SeqComb);
}

#endif

