/*******************************************************
    Global pairwise alignment using dynamic programming approach
    The algorithm of Needleman and Wunsch is used

    The results of global alignment are represented
    as a chain of fragments.
    
    Example:
    
    Seq A = GCATCTGCAGAATCTC
    Seq B = GCAACT ATGGCTGAC
    
    The result is represented as
    
    (BLK 3) -> (POI 1) -> (BLK 2) -> (DEL 1) ->
    (POI 2) -> (BLK 1) -> (POI 2) -> (BLK 1) ->
    (POI 2) -> (BLK 1)
      
 *******************************************************/

#ifndef GLOBALALIGN
#define GLOBALALIGN

#include <iostream.h>                                                                                                                                          
#include <malloc.h>
#include "matrix.cpp"
#include "maxmin.cpp"

#define INS       1   // Insertion
#define DEL       2   // Deletion
#define POI       0   // Including point mutations only
#define BLK       3   // Block of matches
#define NON       4

#define MAXN_GLO  3000   // maximum length of subsequence that Needleman and Wunsch's method can deal with





class FragmentNode{
  public:

  int type;
  int length;
  FragmentNode *pNext;
  
  FragmentNode(int a, int b, FragmentNode *c);
};

FragmentNode::FragmentNode(int a, int b, FragmentNode *c){
  type = a;
  length = b;
  pNext = c;
}









class FragmentChain{
  public:
  
  FragmentNode *head;
  
  FragmentChain();
  ~FragmentChain();
  void FreeFragment();
  void insert(int a, int b);
  void print();
};

FragmentChain::FragmentChain(){
  head = NULL;
}

FragmentChain::~FragmentChain(){
  FreeFragment();
}

void FragmentChain::FreeFragment(){
  FragmentNode *current;

  for(;head!=NULL;){
    current=head;
    head=head->pNext;
    delete current;
  }
  head = NULL;
}

void FragmentChain::insert(int a, int b){
  head = new FragmentNode(a,b,head);
}

void FragmentChain::print(){
  FragmentNode *current;
  for(current = head;current!=NULL;current=current->pNext){
    switch(current->type){
      case INS: cout<<"INS length: "<<current->length<<endl; break;
      case DEL: cout<<"DEL length: "<<current->length<<endl; break;
      case POI: cout<<"POI length: "<<current->length<<endl; break;
      case BLK: cout<<"BLK length: "<<current->length<<endl; break;
    }
  }  
}






// Score and penalty
int gaps=-6;    // Start penalty for indel events.
int gapc=-1;    // Continue penalty for indel events.
int mismatch=0; // penalty for mismatch.
int match=5;    // score for match.


class GlobalAlignment{
  public:
  
  GlobalAlignment();
  ~GlobalAlignment();

  void GlobalAlign(char *input1, char *input2, int n1, int n2, FragmentChain *result);

  private:
  
  int **score, **Nij, **Wij;
};

GlobalAlignment::GlobalAlignment(){
  score=dmatrix<int>(MAXN_GLO+1,MAXN_GLO+1);
  Wij=dmatrix<int>(MAXN_GLO+1,MAXN_GLO+1);
  Nij=dmatrix<int>(MAXN_GLO+1,MAXN_GLO+1);
}

GlobalAlignment::~GlobalAlignment(){
  free_dmatrix(score);
  free_dmatrix(Nij);
  free_dmatrix(Wij);
}


void GlobalAlignment::GlobalAlign(char *input1, char *input2, int n1, int n2, FragmentChain *result){
  int i,j,old_type,new_type=NON;
  int s,numcomp[4];

  // Initialize the score matrix.
  score[0][0]=0;
  for(i=1;i<=n1;i++)
    score[i][0]=Wij[i][0]=gaps+(i-1)*gapc;
  for(j=1;j<=n2;j++)
    score[0][j]=Nij[0][j]=gaps+(j-1)*gapc;
  
  // Calculate the score of the alignment.
  for(j=1;j<=n2;j++)
    for(i=1;i<=n1;i++){
      s=(input1[i-1]==input2[j-1])?match:mismatch;
      numcomp[DEL]=max(Nij[i-1][j]+gapc,score[i-1][j]+gaps);
      numcomp[INS]=max(Wij[i][j-1]+gapc,score[i][j-1]+gaps);
      numcomp[POI]=score[i-1][j-1]+s;
      Nij[i][j]=numcomp[DEL];
      Wij[i][j]=numcomp[INS];
      score[i][j]=numcomp[which_max(3,numcomp)];
    }
  // Perform backtracking.
  i=n1;
  j=n2;
  do{
    old_type = new_type;
    s=(input1[i-1]==input2[j-1])?match:mismatch;
    numcomp[DEL]=max(Nij[i-1][j]+gapc,score[i-1][j]+gaps);
    numcomp[INS]=max(Wij[i][j-1]+gapc,score[i][j-1]+gaps);
    numcomp[POI]=score[i-1][j-1]+s;
    new_type = which_max(3,numcomp);
    if((new_type==POI)&&(input1[i-1]==input2[j-1]))
      new_type=BLK;

    if(new_type==old_type)
      result->head->length++;
    else if((old_type==DEL)&&(score[i+1][j]-score[i][j]!=gaps)){
        new_type = DEL;
        result->head->length++;
    }else if((old_type==INS)&&(score[i][j+1]-score[i][j]!=gaps)){
        new_type = INS;
        result->head->length++;
    }else
      result->insert(new_type,1);

    switch(new_type){
      case INS: j--; break;
      case DEL: i--; break;
      case POI:
      case BLK: i--; j--; break;
    }
  }while((i>0)&&(j>0));
  if(i>0)
    result->insert(DEL,i);
  if(j>0)
    result->insert(INS,j);
}

#endif
