/* Definition of Robinson and Fould distance:
     See Robinson and Fould (1979) Comparison of weighted labelled trees, in Combinatorial Mathematics VI, LNM 748.

   Usage of the variable bCleared:
     Assign labels from 1 to n to each species in tree 1 with 0 on
     the leftmost and n on the rightmost. bCleared on a node of
     tree 2 is the smallest label of the previously visited species
     under this node and bCleared is defined 0 if none of the species
     under the node has been visited. */

void init_tree_comp(phylonode *root, int height){
  root->height = height;
  root->bCleared = 0;
  root->match_with = NULL;
  if(root->child1!=NULL)
    init_tree_comp(root->child1,height+1);
  if(root->child2!=NULL)
    init_tree_comp(root->child2,height+1);
}

// Search the most recent common ancestor of A and B
phylonode *get_MRCA(phylonode *A, phylonode *B){
  int i;
  phylonode *currentA, *currentB;
  
  currentA = A;
  currentB = B;
  if(B->height > A->height){
    for(i=0;i<B->height-A->height;i++)
      currentB = currentB->parent;
  }else if(B->height < A->height){
    for(i=0;i<A->height-B->height;i++)
      currentA = currentA->parent;
  }
  while(currentA!=currentB){
    currentA = currentA->parent;
    currentB = currentB->parent;
  }
  return currentA;
}

// Compare edge1 in tree1 with tree2
// Note: edge1 divide the species into two groups, Gp1 and Gp2.
// The MCRA of Gp1 in tree 2 is returned.
// root1 refers to the child side of edge1.

phylonode *tree_comp(familytree *tree1, familytree *tree2, phylonode *root1, int *n_visited){
  phylonode *match, *match_ch1, *match_ch2, *current;
  int n_prev_visited;

  n_prev_visited = *n_visited;

  if(root1->child1!=NULL)
    match_ch1 = tree_comp(tree1,tree2,root1->child1,n_visited);
  if(root1->child2!=NULL)
    match_ch2 = tree_comp(tree1,tree2,root1->child2,n_visited);
  if(root1->seq_id!=-1){
    match = tree2->table[root1->seq_id];
    match->match_with = root1;
//    printf("Begin to clear %d %d\n",match->seq_id,n_prev_visited);
    match->bCleared = (*n_visited)+1;
//    printf("Node at height %d is cleared\n",match->height);
    for(current=match->parent;current!=NULL;current=current->parent){
      if(current==tree2->par_root.child1)
        break;
      if((current->child1->bCleared)&&(current->child2->bCleared)){
        current->bCleared = min(current->child1->bCleared,current->child2->bCleared);
//        printf("Node at height %d is cleared\n",current->height);
      }else
        break;
    }
//    printf("Finish clearing\n");
    *n_visited = (*n_visited)+1;
  }else{
    match = get_MRCA(match_ch1,match_ch2);
    if(match->bCleared==n_prev_visited+1){
      match->match_with = root1;
//      printf("Match at height %d %d\n",match->height,n_prev_visited);
    }else{
//      printf("Not match %d %d\n",match->height,n_prev_visited);
    }
  }
  return match;
}

int find_weight(int n_markers){
  return n_markers;
}

int find_weight0(int n_markers){
  return (n_markers!=0);
}

int Robinson_Fould12(phylonode *root2, int (*func)(int) ){
  int temp;
//  printf("Checking %d\n",root2->seq_id);
  if(root2->match_with==NULL){
    temp = func(root2->n_markers);
//    printf("%d no match temp = %d\n",root2->seq_id,temp);
  }else{
    temp = abs(func(root2->n_markers)-func(root2->match_with->n_markers))-func(root2->match_with->n_markers);
//    printf("%d match with %d temp = %d\n",root2->seq_id,root2->match_with->seq_id,temp);
  }
  if(root2->child1!=NULL)
    temp += Robinson_Fould12(root2->child1,func);
  if(root2->child2!=NULL)
    temp += Robinson_Fould12(root2->child2,func);
  return temp;
}

int Robinson_Fould1(phylonode *root1, int (*func)(int) ){
  int temp;
  temp = func(root1->n_markers);
  if(root1->child1!=NULL)
    temp += Robinson_Fould1(root1->child1,func);
  if(root1->child2!=NULL)
    temp += Robinson_Fould1(root1->child2,func);
  return temp;
}

int Robinson_Fould(familytree *tree1, familytree *tree2){
  int n_visited = 0;
  init_tree_comp(tree2->par_root.child1,0);
  tree_comp(tree1,tree2,tree1->par_root.child1,&n_visited);
  return Robinson_Fould1(tree1->par_root.child1,find_weight)+Robinson_Fould12(tree2->par_root.child1,find_weight);
}

int Robinson_Fould0(familytree *tree1, familytree *tree2){
  int n_visited = 0;
  init_tree_comp(tree2->par_root.child1,0);
  tree_comp(tree1,tree2,tree1->par_root.child1,&n_visited);
  return Robinson_Fould1(tree1->par_root.child1,find_weight0)+Robinson_Fould12(tree2->par_root.child1,find_weight0);
}

