%This program implements the single-frame complex wave retrieval algorithm in the paper 
%G. Luo, Y. He, X. Shu, R. Zhou, and T. Blu, "Complex Wave and Phase Retrieval
%from A Single Off-Axis Interferogram". JOSA A. Accepted (2022).

%change parameter 'noisePNSR' to 10, one can obtain results in Fig.S1 (in supplemental document 1)
clear;close all;clc;
addpath('Utilities/');
addpath('Functions/');

% Procedures:
% 1, Key parameters
% 2, Object generation
% 3, Reference, interferogram generations
% 4, Complex wave retrieval

% varibles in data generation:
%   'UO' - object wave
%   'UR' - reference wave
%   'In' - interferogram
% varibles of output:
%   'A_FT'  - estimated object wave by using Fourier filtering method
%   'UOest' - estimated object wave by using our method

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%            Key parameters          %%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
M = 512; N = 512;       %image dimension
rough_amp_Ratio = [0.7];% amplitude Ratio
NO_0 = 1.3;             % numerical overlap
rho= sqrt(2*pi)/3;      % object wave bandwidth = pi*rho^2 for a circular support;
epsilon=1*(2*pi/M);     % reference wave amplitude bandwidth = pi*epsilon^2 for a circular support;

theta = 45/180*pi;      % theta controls the orientation of fringes
d = 3.0*rho./(2*NO_0 + 1);
fx = floor( d./(2*pi/M).*cos(theta) );
fy = floor( d./(2*pi/M).*sin(theta) );
w=[fx*2*pi/M,fy*2*pi/N];% spatial frequency of the reference wave

noisePNSR = [30];       % noise level (evaluated PSNR) in acqusitions

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%            Object generation          %%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
option = 'amp-c, ph-spoke';
switch option
     case 'amp-c, ph-spoke'
        s=M;%dimension, e.g. 512x512
        n=48; 
        [x,y] = meshgrid((-s/2:1:s/2-1)/(s/2),(-s/2:1:s/2-1)/(s/2));
        t=(atan2(y,x)*n);
        r=x.^2+y.^2;
        Spo = pi/8*sin(t) +pi/3 ;
        Spo(r<0.005|r>2.0)=0;
        img_amp = 255;
        
        ph = double( Spo/max(Spo(:))-0 >0.725 );
        ph(x.^2+y.^2>0.4^2 & x.^2+y.^2<0.45^2 ) = 0.9;
        ph(x.^2+y.^2>0.7^2 & x.^2+y.^2<0.75^2 ) = 0.9;
        UO = img_amp.*exp(2.5*1i*ph/max(ph(:))-2.0);
        clear s n x y t r Spo mask;
end
%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%       Reference and Interferogram        %%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
x=repmat((1:M)',1,N);x=(x-(M+1)/2);
y=repmat(1:N,M,1);y=(y-(N+1)/2);
VR=exp(1i*w(1)*x+1i*w(2)*y);
wrap=@(w)-mod(-w+pi,2*pi)+pi;

filtUO=@(I)imagefilter(I.*conj(VR),@(wx,wy)double(wrap(wx+w(1)).^2+wrap(wy+w(2)).^2<=rho^2),1).*VR;%object wave bandlimiation
filtAR=@(I)real( imagefilter(I.*conj(VR),@(wx,wy)double(wx.^2+wy.^2<=epsilon^2),1) ).*VR;%reference wave bandlimiation

UO=filtUO(UO); %object wave bandlimiation

%constant reference amplitude
UR=1./rough_amp_Ratio.*mean(abs(UO(:))).*VR;    
UR=filtAR(UR); %reference wave bandlimiation

In0=abs(UO+UR).^2;                %noise-free interferogram

[temp] = addnoise(In0,noisePNSR); %add noise in PSNR
temp_noise = temp - In0;
temp_noise(temp - In0<=0) = -temp_noise(temp - In0<=0);%ensure all pixel values are positive
In = In0 + temp_noise; %noisy interferogram  

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%          Complex wave retrieval          %%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Fourier filtering method
tic;
A_FT = ( filtUO(In.*VR) );
t_FT = toc;
A_FT = A_FT*( A_FT(:)\UO(:) );  %rescale.

%Our complex wave retrieval
params.iterMax = 5;
params.input = In;
params.epsilon = epsilon;
params.rho = rho;
params.w = w;
params.VR = VR;
tic;
[URest,UOest]=parafit_interferogram(params);
t_our = toc;
UOest = UOest*( UOest(:)\UO(:) );%rescale. 
%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%          display           %%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fprintf('check PSNR value of data generation\n')
fprintf('noise in interferogram: %.4f dB PSNR\n', PSNR(In,In0) );
NO = (3*rho/norm(w,2)-1)/2;
Ratio = sum(abs(UO(:)))./sum(abs(UR(:)));
fprintf('numerical overlap=%.4f\n',NO);
fprintf('amplitude ratio=%.4f\n',Ratio);

psnr_real_Our = PSNR((real(UOest)),(real(UO)));    
psnr_real_FT = PSNR((real(A_FT)),(real(UO)));    
%  
fprintf('Reconstruction Accuracy (in PSNR) of real part: FT method, %.4f dB, and Ours (5 iterations), %.4f dB\n',...
    psnr_real_FT,psnr_real_Our );
fprintf('Computational time by FT method, %.2f seconds, and by Ours (5 iterations), %.2f seconds\n',...
    t_FT,t_our );


visualFactor = 1;
aux_interferogram_Fouier_visual(In,rho,w,visualFactor,true(1));

figure,
imview(imcontrast( real(A_FT), real(UOest), real(UO) ))




 