function [phi,q] = qangle(input,fitmat)
%This program implements the phase restoration algorithm in the paper 
%G. Luo, Y. He, X. Shu, R. Zhou, and T. Blu, "Complex Wave and Phase Retrieval
%from A Single Off-Axis Interferogram". JOSA A. Accepted (2022).
% FUNCTION : Finds the "best" quadratic estimate of the phase of the complex image input.
% -----------  
% INPUT:
%      input              -- complex-valued wave or real valued wrapped phase;
%      fitmat             -- polynomials, e.g. Chebshev or Zernike polynomials
% OUTPUT:
%      phi      -- estimated distorted phase
%      q        -- fitted coefficients 
%
% REFERENCES:
%     [1] G. Luo, R. Guo, R. Zhou, and T. Blu, "A Fast Robust Phase 
%         Restoration Algorithm." In Computational Optical Sensing and 
%         Imaging, pp. CM2A-6. Optica Publishing Group, 2022.
%     [2] G. Luo, Y. He, X. Shu, R. Zhou, and T. Blu, "Complex Wave and
%         Phase Retrieval from A Single Off-Axis Interferogram". JOSA A. Accepted (2022).
%
% AUTHOR : Gang Luo and Thierry Blu
% DATE   : November 2022
% CONTACT: Gang Luo (luogang@link.cuhk.edu.hk), The Chinese University of Hong Kong.
%
%
% EXAMPLE OF USE:
% % addpath('Utilities/');
% % addpath('Functions/');
% % M=512;N=512;
% % x=1:M;x=(x-(M+1)/2)/(M-1);x=repmat(x',[1,N]);
% % y=1:N;y=(y-(N+1)/2)/(N-1);y=repmat(y,[M,1]);
% % fitmat=[ones(M*N,1),x(:),y(:),x(:).^2,y(:).^2,x(:).*y(:)];
% % q0=100*randn(6,1);
% % phi0=reshape(fitmat*q0,M,N);
% % A0=exp(i*phi0);
% % A1=addnoise(A0,0);%PSNR 0 dB
% % [phi,q]=qangle(A1,fitmat);
% % disp(['Maximum phase difference between estimate and noiseless : ' num2str(max(abs(mod(phi(:)-phi0(:)+pi,2*pi)-pi)))])
% % figure,imview(imcontrast( angle(A0),angle(A1),angle(exp(i*phi)) ))
%

[M,N]=size(input);

if isreal(input)% input is the wrapped phase
    input = exp(1j*input);% complex value
end
    
fit=@(phi,mask)reshape(fitmat*(fitmat(mask(:),:)\phi(mask(:))),M,N);
phi=0*input;
encore=1;
k=0;kmax=100;
thres = pi/2;
while encore&k<kmax
    % find the largest connected region for which the phase is within pi/2
    % of the most frequent phase value   
    alpha=angle(input.*exp(-i*phi));
    mask=bwareafilt(abs(alpha)<thres,1);
    dphi=fit(alpha,mask);            % fitting a quadratic expression only to that connected region
    phi=phi+dphi;                    % phase update
    encore=(max(abs(dphi(:)))>1e-3);% e.g.1e-10        
    k=k+1;
end
q=lsqminnorm(fitmat,phi(:),1e-6);
% disp(['Number of iterations : ' num2str(k)])
  
end