function [deblurred,varargout]=iLET_deconv(blurred,PSF,varargin)
%[deblurred,varargout]=ILET_DECONV(blurred,PSF,varargin)
% 
% Iterative LET restoration with decimated/undecimated wavelet transform. 
% Here the LET basis consist of:
% 
% 	1) previous two iterate results
%  	2) soft-threshold
%  	3) pre-conditioned generalized gradient
% 
% See the references for details.
% 
% =========================================================================
% 
% REQUIRED INPUT
% 
%   blurred: blurrend and noisy measurment
% 
%   PSF: centered convolution kernel
% 
% =========================================================================
% 
% OPTIONAL INPUT
% 
%  	W: function handeler -- wavalet synthesis operator
% 
%  	WT: function handeler -- wavelet anslysis operator
% 
%  	LAMBDA: regularization weights
% 
%   tau: positive step-size in the soft-threshold function
% 
%   showValue: whether to output the objective function value during 
%       iterations. Default 0, i.e. do not show objective function values.
% 
%	initialization: initialization method. Default 1.
%       1) initialization = 1, use Tikhonov regularized inverse
%       2) initialization = 2, use random values
% 
%	maxiter: maximum iteration number. Default 30.
% 
%	innermax: maximum inner IRLSloop iteration number. Default 5.
% 
%	refSol: reference solution, i.e. the global minimizer of the
%               ell-1 regularized objective function.
% 
%	stopCri: criterion used to terminate iteration. Default 0.
%       1) stopCri = 0, terminate when maximum iteration allowed is reached.
%       2) stopCri = 1, terminate when the reconstruction is close enough 
%       (defined by 'tol') to the reference solution.
% 
%   tol: measures how 'close' the reconstruction should be to the reference
%       solution when 'stopCri' = 1. Default 40[dB].
% 
%	updateReg: wehther update the ell-1 regularization weight or not. 
%       Default 0, i.e. not update the weight.
% 
%	n_eng: noise energy.
% 
% =========================================================================
% 
% OUTPUT
% 
%   deblurred: deblurred image
% 
%   varargout: with the order as follows
%       varargout{1} = count: iteration numbers
%       varargout{2} = obj: ell_1 regularized objecitve function values
%       varargout{3} = lambda_final: the updated regularization weight if 
%                       opt.updateReg = 1
%       varargout{4} = quadVal: values for the quadratic part in the 
%                       objective function: |y-Hx|^2
%       varargout{5} = lambdas: regularization weights throughout 
%                       iterations if the option 'updateReg' = 1.
% 
% =========================================================================
% 
% Authors: Hanjie Pan and Thierry Blu, October 2011
% Image and Video Processing Lab, CUHK, Shatin, Hong Kong
% This software is freely available at http://www.ee.cuhk.edu.hk/~hjpan
% 
% References:
% [1] H. Pan and T. Blu, "Sparse image restoration using iterated linear 
% expansion of thresholds," in 2011 IEEE International Conference on Image 
% Processing (ICIP2011), (Brussels, Belgium), September 2011.
% 
% [2] H. Pan and T. Blu, "An iterated linear expansion of thresholds for 
% l1-based image restoration." Submitted to IEEE Transactions on Image 
% Processing.
%
warning('off','MATLAB:nearlySingularMatrix');
% set default options

% wavelet analysis deapth
wDeapth=3;

% wavelet scaling filter, here it is symlet 8
F_scal=daubcqf(16,'mid');

% wavelet synthesis operator
W=@(c) midwt(c,F_scal,wDeapth);

% wavelet analysis operator
WT=@(x) mdwt(x,F_scal,wDeapth);

% whether output intermediate results or not
showValue=0;

% initialization of the iterative algorithm
%	1) initialization = 1, use Tikhonov regularized inverse
%	2) initialization = 2, use random values
initialization=1;

% maximum i-LET iterations
maxiter=5;

% maximum IRLS inner loop iterations
innermax=5;

% criterion used to terminate i-LET iteration. Default 0.
%	1) stopCri = 0, terminate when maximum iteration allowed is reached.
%	2) stopCri = 1, terminate when the reconstruction is close enough 
%	(defined by 'tol') to the reference solution.
stopCri=0;

% tol: measures how 'close' the reconstruction should be to the reference
% solution when 'stopCri' = 1. Default 40[dB].
if stopCri
    tol=40;
end

% seed used when random initialization is used such that we have
% reproducible results
seed=round(abs(1e7*randn(2,1)));

% updateReg: wehther update the ell-1 regularization weight or not. Default
% 1, i.e. update the weight.
updateReg=1;

% size of the blurred measurement
[N,M]=size(blurred);
NM=N*M;N2inv=1/NM;

% estimate noise level from the blurred measurement
sig2_est=(estimate_noise(blurred))^2;
n_eng=sig2_est*NM;

% ell-1 regularization weight
lambda=0.0745*sig2_est+0.69;

% estimate max pixel values of the image in spatial domain, e.g. 255 for 
% 8-bit gray-scale image; 65535 for 16-bit gray-scale image.
xmax=2^(nextpow2(max(blurred(:))-3*sqrt(sig2_est)))-1;

% maximum possible wavelet coefficients value
cmax=xmax*sum(F_scal(F_scal(:)>=0));
% soft-threshold step-size
tau=cmax/lambda;

if nargin<2
	error('The i-LET algorithm requires both the measurements and the PSF.\n');
% parse optional options
elseif (rem(length(varargin),2)==1)
	error('Optional parameters should be provided in pairs');
else
    for i=1:2:(length(varargin)-1)
        switch upper(varargin{i})
            case 'W'
                W=varargin{i+1};
            case 'WT'
                WT=varargin{i+1};
            case 'LAMBDA'
                lambda=varargin{i+1};
                tau=cmax/lambda;
            case 'TAU'
                tau=varargin{i+1};
            case 'SHOWVALUE'
                showValue=varargin{i+1};
            case 'INITIALIZATION'
                initialization=varargin{i+1};
            case 'INNERMAX'
                innermax=varargin{i+1};
            case 'STOPCRI'
                stopCri=varargin{i+1};
                if stopCri
                    tol=40;
                    maxiter=1000;
                end
            case 'TOL'
                tol=varargin{i+1};
            case 'MAXITER'
                maxiter=varargin{i+1};
            case 'SEED'
                seed=varargin{i+1};
            case 'REFSOL'
                refSol=varargin{i+1};
            case 'N_ENG'
                n_eng=varargin{i+1};
            case 'UPDATEREG'
                updateReg=varargin{i+1};
            otherwise
                error(['Unrecognized option: ''' varargin{i} '''']);
        end
    end
end
% check the optional input arguments
if stopCri && ~exist('refSol','var')
    % if stopCri = 1, i.e. measured by the closeness to the ell_1 solution
    % at convergence, then the reference solution (refSol) has to be 
    % provided as well.
    error('iLET:argChk','\nPLEASE PROVIDE THE L_1 REFERENCE SOLUTION TO COMPARE WITH.\n');
end

Y=fftn(blurred);
H=fftn(PSF);
Hc=conj(H);
HcH=Hc.*H;
HcY=Hc.*Y;
Y=Y(:);
a=[1;10]./tau;
HcHinv1=HcH./(HcH+a(1));
HcHinv2=HcH./(HcH+a(2));
% Initialization
% ---------------------------------
switch initialization
    case 1
    % initialize the reconstructed image with an Tikhonov regularized
    % inverse: x=(H^T*H+mu Id)^{-1}(H^T*y)
        c1=1/(0.05*lambda).*WT(real(ifftn((1-HcHinv1).*HcY)));
        c2=1/(0.005*lambda).*WT(real(ifftn((1-HcHinv2).*HcY)));
        x1=W(c1);
        x2=W(c2);
        x1_ft=fftn(x1);
        x2_ft=fftn(x2);
        cPrev=[c1(:) c2(:)];
    case 2
    % initialize the wavelet coefficients of reconstructed image with 
    % random values
        randn('state',seed);% to have reproducible results
        c1=randn(size(WT(blurred)));
        c2=randn(size(c1));
        x1=W(c1);
        x2=W(c2);
        x1_ft=fftn(x1);
        x2_ft=fftn(x2);
        cPrev=[c1(:) c2(:)];
    otherwise
        error('Unrecognized initialization option.');
end
Hx_prev1=H.*x1_ft;
Hx_prev2=H.*x2_ft;
Hx_prev=[Hx_prev1(:),Hx_prev2(:)];
xPrev=[x1(:),x2(:)];

c_recon=c1;
deblurred=x1;
x_recon_hat=x1_ft;

% determine whether decimated or undecimated wavelet transform is used
if size(c_recon,1)<size(c_recon,2)
    flag='rwt';
else
    flag='dwt';
end
% % ------------------------------------- % %
if showValue
    fprintf('#iter     fun-val\n');
    fprintf('==================\n');
end
obj=[];
count=0;
dim=5;
F=zeros(numel(c_recon),dim);
HWF=complex(zeros(NM,dim),0);
FtDF=zeros(dim);
G=zeros(numel(c_recon),dim*(dim+1)/2);
lambdas=[];
% Depending on the transformation type, the low-low subband will be used
% (redundant wavelet transform) or not (for decimated wavelet transform)
if strcmpi(flag,'rwt')
    lambda_n=lambda;
else
    % only non-ll subbands are used in regularization
    mask=ones(N,M);
    mask(1:N/(2^wDeapth),1:M/(2^wDeapth))=0;
    lambda_n=lambda.*mask;
end
% soft threshold level
th_Level=0.5*tau.*lambda_n;
% Iterative LET restoration
% ---------------------------------
while 1
    count=count+1;
    % % ====================build LET basis====================== % %
    % grad2: H^t(Hx-y)
    grad2=WT(ifft2(x_recon_hat.*HcH-HcY,'symmetric'));
    Th_Wtbase=soft(c_recon-tau.*grad2,th_Level);
    % u: the 'generalized gradient' in wavelet domain
    % Wu: in image domain; Wu_ft: in Fourier domain; Hx: convolution with
    % filter 'H' in Fourier domain
    u=c_recon-Th_Wtbase;
    x=W(u);
    x_ft=fftn(x);
    HWu=x_ft.*H;
    % u1, u2: (A'*A+mu*Id)^{-1}*Wu
    Wu1_ft=x_ft.*HcHinv1;
    Wu2_ft=x_ft.*HcHinv2;
    Wu1_Wu2=ifftn(Wu1_ft+1j.*Wu2_ft);
    Wu1=real(Wu1_Wu2);
    Wu2=imag(Wu1_Wu2);
    u1=u-WT(Wu1);
    u2=u-WT(Wu2);
    HWu1=(x_ft-Wu1_ft).*H;
    HWu2=(x_ft-Wu2_ft).*H;
    % rearrange basis in matrix form
    F(:,1:2)=cPrev;F(:,3)=u(:);F(:,4)=u1(:);F(:,5)=u2(:);
    HWF(:,1:2)=Hx_prev;HWF(:,3)=HWu(:);HWF(:,4)=HWu1(:);HWF(:,5)=HWu2(:);
    
    % % ================solve LET coefficients=================== % %
    % initialize LET coefficients
    D=lambda_n(:)./(abs(c_recon(:))+1e-6);
    HF_t_HF=real(HWF'*HWF).*N2inv;
    HF_t_y=real(HWF'*Y).*N2inv;
    % pre-compute G for the consideration of efficiency
    n=0;
    for l=1:dim
        for m=l:dim
            n=n+1;
            G(:,n)=F(:,l).*F(:,m);
        end
    end
    
    for i=1:innermax
        GD=G'*D;
        n=0;
        for l=1:dim
            for m=l:dim
                n=n+1;
                FtDF(l,m)=GD(n);
                FtDF(m,l)=FtDF(l,m);
            end
        end
        M=HF_t_HF+0.5*FtDF;
        A=M\HF_t_y;
        c_recon=F*A;
        D=lambda_n(:)./max(abs(c_recon),1e-15);
    end
    
    % % ========================================================= % %
    % reconstructed image in spatial domain
    c_recon=reshape(c_recon,N,[]);
    deblurred=W(c_recon);
    switch stopCri
        case 0
            stop=(count>maxiter);
        case 1
            stop=(PSNR(deblurred,refSol,255)>tol);
        otherwise
            error('Non-reognized stopping criterion.');
    end
    
    if stop
        break;
    end
    
    % % --------update LET basis---------- % %
    x_recon_hat=fftn(deblurred);
    Hx_recon_hat=H.*x_recon_hat;
    xPrev(:,2)=xPrev(:,1);
    cPrev(:,2)=cPrev(:,1);
    Hx_prev(:,2)=Hx_prev(:,1);
    xPrev(:,1)=deblurred(:);
    cPrev(:,1)=c_recon(:);
    Hx_prev(:,1)=Hx_recon_hat(:);
    if showValue || updateReg || nargout>2
        obj_quad=norm(Y-Hx_recon_hat(:),2)^2.*N2inv;
        quadVal(count)=obj_quad;
        obj=[obj,obj_quad+norm(lambda_n(:).*c_recon(:),1)];
    end    
    if showValue
        fprintf('%3d %15.5e\n',count,obj(end));
    end
    if updateReg
        lambda_n=n_eng/obj_quad*lambda_n;
        lambdas=[lambdas,lambda_n(end)];
        th_Level=0.5*tau.*lambda_n;
    end
end
lambda_final=lambda_n(end);
% parse output arguments
out_opt={'count;','obj;','lambda_final;','quadVal;','lambdas;'};
for i=1:nargout-1
    varargout{i}=eval(out_opt{i});
end
return
