%ILET_DEMO2.M 
% Deconvolution by minimizing the l1-regularized objective function J(c):
% 
% 	min_c   |y-HWc| + lambda |c|_1
% 
% with wavelet transform. Here 'H' is the PSF convolution matrix, 'W' is 
% the wavelet synthesis operator, and 'c' are the wavelet coefficients such
% that the spatial domain image x = Wc.
% 
% Authors: Hanjie Pan and Thierry Blu, October 2011
% Image and Video Processing Lab, CUHK, Shatin, N.T., Hong Kong
% This software is available at http://www.ee.cuhk.edu.hk/~hjpan
% 
% References:
% 
% [1] H. Pan and T. Blu, "Sparse image restoration using iterated linear 
% expansion of thresholds," in 2011 IEEE International Conference on Image 
% Processing (ICIP2011), (Brussels, Belgium), September 2011.
% 
% [2] H. Pan and T. Blu, "An iterated linear expansion of thresholds for 
% l1-based image restoration." Submitted to IEEE Transactions on Image 
% Processing.
% 
clear;close all;%clc;
dir=fileparts(mfilename('fullpath'));
addpath([dir,'/src'])              	% source files for all three comparing algorithms
addpath(genpath([dir,'/tools']))   	% 'tools' include rice wavelet toolbox
addpath([dir,'/image'])            	% where the test images are stored
addpath([dir,'/data'])              % csv files that specify the ell_1 regularization weights used

%% ------- parameters -------
im_name='peppers.tif';           	% test image name

blur_type=2;                       	% blur type --- 1: 9 x 9 uniform blur; 
                                  	% 2: 1/(1+i^2+j^2) for i,j=-7:7; 
                                    % 3: [1 4 6 4 1]^T [1 4 6 4 1];
                                    % 4: Customized PSF by loading a *.mat file

BSNR_db=40;                        	% blurred signal to noise ratio (in dB)

%% Distort (blur and add noise) original image
im=double(imread(im_name));         % input image
[N,M]=size(im);                     % image size

% blurring
PSF=convKernel([N,M],blur_type);   	% create convolution kernel
                                    % Here 'PSF' is zero padded and centered at the 
                                    % upper-left corner of the image
y=convolve2(im,PSF);
sigma=norm(y-mean(y(:)),'fro')/sqrt(N*M*10^(0.1*BSNR_db));
randn('state',[362436069;521288629]);% to have reproducible results
% blurred and noisy measurement
noise=sigma.*randn(N,M);
blurred=convolve2(im,PSF)+noise;

psnr_0=PSNR(blurred,im);%

%% -- i-LET deconvolution --
wDeapth=3;                              % wavelet analysis deapth
F_scal=daubcqf(16,'mid');               % wavelet scaling filter
% WT=@(x) w_ana2D_rwt(x,F_scal,wDeapth);  % wavalet analysis operator
% W=@(c) w_syn2D_rwt(c,F_scal,wDeapth);   % wavalet synthesis operator
W=@(c) midwt(c,F_scal,wDeapth);
WT=@(x) mdwt(x,F_scal,wDeapth);

% -- load reference solution -- %
[~,basename]=fileparts(im_name);
if size(WT(blurred),1)<size(WT(blurred),2)
    transformType='rwt';
else
    transformType='dwt';
end
filename=['ref_sol/',transformType,'/',basename,'/',basename,'_BSNR_',...
    num2str(BSNR_db),'_type_',num2str(blur_type),'_blur.mat'];
if ~exist([dir,'/',filename],'file')
    fprintf('Reference solution not found!\n');
    wwwHome='http://www.ee.cuhk.edu.hk/~hjpan';
	getSolOptions=['[1] Download reference solution from ',wwwHome,'\n',...
        '[0] None.\nChoice: '];
    response=input(getSolOptions,'s');
	switch response
        case '1'
            if ~exist([dir,'/ref_sol/',transformType,'/',basename],'dir')
                mkdir([dir,'/ref_sol/',transformType,'/',basename]);
            end
            urlwrite([wwwHome,'/ilet/',filename],filename);
        otherwise
            break;
	end
end
load([dir,'/',filename]);

% -- read the ell_1 regularization weight -- %
csv_filename=[basename,'_',transformType,'_reg_',num2str(wDeapth),'level.csv'];
lambda=csvread(csv_filename,blur_type-1,BSNR_db/5-2,...
    [blur_type-1,BSNR_db/5-2,blur_type-1,BSNR_db/5-2]);

t0=tic;
% 
% % CASE I: stop when maximum iterations are reached
% [deblurred,count,obj]=iLET_deconv(blurred,PSF,...
%                     'W',W,'WT',WT,'lambda',lambda,'initialization',1,...
%                     'updatereg',0,'stopcri',0,'maxiter',100,...
%                     'refsol',x_recon_ref);
% 
% ======================================================================= %
% 
% CASE II: stop when the reconstructions are close enough to the reference
% solution.
[deblurred,count]=iLET_deconv(blurred,PSF,...
                    'W',W,'WT',WT,'lambda',lambda,'initialization',1,...
                    'updatereg',0,'stopcri',1,'refsol',x_recon_ref);
t_ilet=toc(t0);

fprintf('Finished!\n');
psnr_ilet=PSNR(deblurred,im);

%% ------- plot results -------
h1=figure(1);
set(h1,'name','i-LET Unconstrained Formulation Demo');
set(h1,'Units','normalized');
set(h1,'Position',[0,0.3,1,0.5]);
xmax=2^(nextpow2(max(y(:))))-1;
subplot(131),imagesc(im,[0 xmax]);
axis image;axis off;colormap gray;
title('{\bf Original}','FontSize',14,'interpreter','latex')
subplot(132),imagesc(blurred,[0 xmax]);
axis image;axis off;colormap gray;
title(['{\bf Blurred: PSNR = ',num2str(psnr_0,'%.2f'),'dB}'],...
    'FontSize',14,'interpreter','latex')
subplot(133),imagesc(deblurred,[0 xmax]);
axis image;axis off;colormap gray;
title(['{\bf Deblurred: PSNR = ',num2str(psnr_ilet,'%.2f'),'dB}'],...
    'FontSize',14,'interpreter','latex')
fprintf('Input PSNR  : %.2f[dB]\n',psnr_0);
fprintf('Output PSNR : %.2f[dB]\n',psnr_ilet);
fprintf('Time Elapsed: %5.2f[sec]\n',t_ilet);