function [G,H]=getGH(G,p,q);
%%%
%%% TITLE: getGH
%%%
%%% PROVIDES: the low-pass and high-pass orthonormal filters
%%% of a rational (p/q,p/(p-q)) filterbank, given the low-pass.
%%%
%%% USAGE: [G,H]=getGH(G,p,q);
%%%
%%% NOTE: G is then recomputed according to the factorization 
%%% of the lossless matrix of the filterbank. This procedure requires
%%% a very low orthonormality error of the low-pass filter. Another requirement 
%%% is that the degree of G be of the form k*p-1.
%%% If p-q>=2 there are several different non-trivial solutions for the
%%% high-pass filter H. This is reflected in function "ortho" which chooses
%%% one of them by a random procedure.
%%%
%%% REFERENCE: T. Blu, "A new design algorithm for two-band orthonormal rational 
%%% filter banks and orthonormal rational wavelets," IEEE Trans. Signal Processing, 
%%% vol. 46, no. 6, pp. 1494-1504, June 1998.
%%%
%%% AUTHOR: Thierry BLU, Swiss Federal Institute, Lausanne (EPFL)
%%%
%%% VERSION: 1993-1998
%%%

L=length(G);
M=rat_mat(G,p,q);
[R0,R1]=inv_lossless(M,p);
M=lossless(ortho(R0),R1);
[H,dH]=inv_rat_mat(M(q+1:p,:),p,p-q);
if dH<0,
	H=[H zeros(1,dH+p*ceil(-dH/p))];
end
[G,dG]=inv_rat_mat(M(1:q,:),p,q);
if L~=length(G)
  disp('roundoff error')
end
if nargout==1,G=H;end

%%%
%%% END OF THE MAIN PROGRAM
%%%

%%%
%%% OTHER FUNCTIONS
%%%

function M=rat_mat(G,p,q)
G=rev(G);
M=[];
M(q,p*(floor((length(G)-q)/p/q)+2))=0;
for k=0:q-1
  for n0=0:p-1
    for n1=0:length(M(1,:))/p-1
      a=k*p+q*(n1*p-n0)+1;
      if a>=1&a<=length(G)
        M(k+1,n0+n1*p+1)=G(a);
      end
    end
  end
end
    
%%%
%%%

function [G,dmin]=inv_rat_mat(M,p,q)
epsilon=1e-8;
[q,N]=size(M);
dmin=-q*(p-1);
G=[];
for k=0:q-1
  for n0=0:p-1
    for n1=0:N/p-1
      a=k*p+q*(n1*p-n0)+1;
      G(a-dmin)=M(k+1,n0+n1*p+1);
    end
  end
end
n=length(G);
encore=(abs(G(n))<=epsilon);
while encore
  n=n-1;
  encore=(abs(G(n))<=epsilon);
end
G=G(1:n);
G=rev(G);
n=length(G);
encore=(abs(G(n))<=epsilon);
while encore
  dmin=dmin+1;
  n=n-1;
  encore=(abs(G(n))<=epsilon);
end
G=G(1:n);

%%%
%%%

function G=lossless(R,M);
[q,p]=size(R);
[x,N]=size(M);
G=R;
for n=1:N,
  [U0,U1]=simple_element(M(:,n));
  G(q,length(G(1,:))+p)=0;
  for nn=length(G(1,:))-p:-p:p
    G(:,nn+1:nn+p)=G(:,nn+1:nn+p)*U0+G(:,nn-p+1:nn)*U1;
  end
  G(:,1:p)=G(:,1:p)*U0;
end

%%%
%%%

function [R,M]=inv_lossless(G,p);
epsilon=1e-11;
n=find(abs(G)<=epsilon);
G(n)=zeros(size(n));
[q,N]=size(G);
N1=floor((N-1)/p+1)*p;
if N<N1
  N=N1;
  G(N)=0;
end
encore=(N>=2*p);
k=0;
x=[];
while encore
  k=k+1;
  U=G(:,N-p+1:N).';
  [a,b]=max(max(abs(U)));
  U=U(:,b);
  U=U/norm(U);
  M(:,k)=U;
  for n=0:p:N-2*p
    Gn=G(:,n+1:n+p);
    Gnn=G(:,n+p+1:n+2*p);
    G(:,n+1:n+p)=Gn+((Gnn-Gn)*U)*U.';
  end
  G(:,N-p+1:N)=G(:,N-p+1:N)-(G(:,N-p+1:N)*U)*U.';
    N=N-p;
  encore=(N>=2*p);
end 
if k>=2
  for k=1:p
    M(k,:)=rev(M(k,:));
  end
end
R=G(:,1:p);

%%%
%%%

function [M0,M1]=simple_element(U);
U=U/norm(U);
M1=U*U.';
M0=eye(length(U))-M1;

%%%
%%%

function R=ortho(V);
[p,q]=size(V);
transposition=0;
if p<q
  V=V.';
  transposition=1;
  [p,q]=size(V);
end
for k=q+1:p
  V(:,k)=rand(p,1);
end
for k=1:p
  R(:,k)=V(:,k)/norm(V(:,k));
  for s=k+1:p
    V(:,s)=V(:,s)-(R(:,k).'*V(:,s))*R(:,k);
  end
end
if transposition
  R=R.';
end

%%%
%%%

function H=rev(G)
H=G(length(G):-1:1);
