function [FFTanalysisfilters,FFTsynthesisfilters]=FFTfractsplinefilters_d(M,alpha,tau,type)
%	Usage: [FFTanalysisfilters,FFTsynthesisfilters]=FFTfractsplinefilters(M,alpha,tau,type) 
%
% 	Provides the frequency response of the low- and high-pass 
% 	filters that generate the orthonormal or semi-orthonormal (B-spline  
%	or dual) fractional splines of degree alpha, shift tau (see ref. [3]) and of given type.
%
% 	See also FFTWAVELETANALYSIS, FFTWAVELETSYNTHESIS.
% 	Uses FRACTSPLINEAUTOCORR
%
% 	Author: Thierry Blu
% 	The Chinese University of Hong Kong.
% 	This software is downloadable at http://www.ee.cuhk.edu.hk/~tblu/demos/
%
%		%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%		%%%%%%%%%%%% INPUT %%%%%%%%%%%%	
%		%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% 	M 	  = size of the input signal = length of FFTfilters = 2^N
% 	alpha = degree of the fractional splines, must be >-0.5 
% 	tau	  = shift or asymmetry of the fractional splines, we suggest to restrict this value to
%		    the interval [-1/2,+1/2] because tau+1 leads to the same wavelet space as tau.
%		    Particular cases
%		 	   tau=0 <=> symmetric splines; 
%			   tau=1/2 <=> max. dissymetric splines
%   		   tau=(alpha+1)/2 <=> causal splines)
% 	type  = type of the B-splines
%		     = 'ortho' (orthonormal, default)
% 		     = 'bspline' (B-spline) 
% 		     = 'dual' (dual). This last option is the flipped version of 'bspline'.
%
%		%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	
%		%%%%%%%%%%%% OUTPUT %%%%%%%%%%%	
%		%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	
%
% 	FFTanalysisfilters	= [lowpassfilter;highpassfilter]	: FFT filter arrays
% 	FFTsynthesisfilters	= [lowpassfilter;highpassfilter] 	: FFT filter arrays
% 
%		%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	
%		%%%%%%%%%%%% REFERENCES %%%%%%%
%		%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	
%
% 	[1] Unser, M. & Blu, T.,"Fractional Splines and Wavelets", SIAM Review, Vol. 42 (1), pp. 43-67, March 2000. 
% 	[2] Blu, T. & Unser, M.,"The Fractional Spline Wavelet Transform: Definition and Implementation", Proc. IEEE International 
%       Conference on Acoustics, Speech, and Signal Processing (ICASSP'00), Istanbul, Turkey, Vol. {I}, pp. 512-515, June 5--9, 2000.  
%   [3] Blu, T. & Unser, M.,"A Complete Family of Scaling Functions: The $($-Fractional Splines", Proc. IEEE International 
%       Conference on Acoustics, Speech, and Signal Processing (ICASSP'03), Hong Kong, China, Vol. {VI}, pp. 421-424, April 6--10, 2003. 

u=alpha/2-tau;
v=alpha/2+tau;

if alpha<=-0.5
	disp('The autocorrelation of the fractional splines exists only ')
	disp('for degrees strictly larger than -0.5!')
	FFTanalysisfilters=[];
	FFTsynthesisfilters=[];
	return
end
if M~=2^round(log(M)/log(2))
	disp(' ')
	disp('The size of the FFT must be a power of two!')
	disp(' ')
	FFTanalysisfilters=[];
	FFTsynthesisfilters=[];
	return
end
nu=0:1/M:(1-1/M);

A=fractsplineautocorr(alpha,nu);
A2=[A A];
A2=A2(1:2:length(A2));		% A2(z) = A(z^2)

if type(1)=='o'|type(1)=='O'
	% orthonormal spline filters
	lowa=sqrt(2)*((1+exp(2*i*pi*nu))/2).^(u+0.5).*((1+exp(-2*i*pi*nu))/2).^(v+0.5).*sqrt(A./A2);
	
	higha=exp(2*i*pi*nu).*lowa;
	higha=conj([higha(M/2+(1:M/2)) higha(1:M/2)]);
	
	lows=lowa;
	highs=higha;
	FFTanalysisfilters=[lowa;higha];
	FFTsynthesisfilters=[lows;highs];
else
	% semi-orthonormal spline filters
	lowa=sqrt(2)*((1+exp(2*i*pi*nu))/2).^(u+0.5).*((1+exp(-2*i*pi*nu))/2).^(v+0.5);
	
	higha=exp(2*i*pi*nu).*lowa.*A;
	higha=conj([higha(M/2+(1:M/2)) higha(1:M/2)]);
	
	lows=lowa.*A./A2;
	highs=higha./(A2.*[A(M/2+(1:M/2)) A(1:M/2)]);
	
	if type(1)=='d'|type(1)=='D' 
		% dual spline wavelets
		FFTanalysisfilters=[lowa;higha];
		FFTsynthesisfilters=[lows;highs];
	else
		% B-spline wavelets
		if type(1)=='b'|type(1)=='B'
			FFTsynthesisfilters=([lowa;higha]);
			FFTanalysisfilters=([lows;highs]);
		else
			error(['''' type '''' ' is an unknown filter type!'])
		end
	end	
end
