function x=FFTwaveletsynthesis1D(w,FFTsynthesisfilters,J);

% 	Usage: x=FFTwaveletsynthesis1D(w,FFTsynthesisfilters,J)
%
%   FFT-based implementation of the inverse wavelet transform (see ref. [2]). It computes the 
%   inverse wavelet transform of "w" using a Fourier method. This function is the inverse of 
% 	FFTWAVELETANALYSIS. It uses periodic boundary conditions. The length of the signal must be 
%   a power of two and the frequency responses of the filters are specified in FFTsynthesisfilters.
%
%   The wavelet coefficient vector has the following fine-to-coarse 
% 	organization: w=[wav1 wav2 ...  wavJ lowJ]
% 
% 	Input:
% 	w = wavelet transform, of size 2^N=length of FFTsynthesisfilters
% 	J = depth of the decomposition, i.e., J wavelet bands + 1 lowpass
% 	FFTsynthesisfilters = [lowpassfilter;highpassfilter]
% 	
% 	Output:
% 	x = signal of size 2^N
% 	
% 	See also FFTWAVELETANALYSIS1D, FFTFRACTSPLINEFILTERS, WEXTRACT1D.
% 	
% 	Author: Thierry Blu
% 	The Chinese University of Hong Kong.
% 	This software is downloadable at http://www.ee.cuhk.edu.hk/~tblu/demos/
% 	
% 	References:
% 	[1] Unser, M. & Blu, T.,"Fractional Splines and Wavelets", SIAM Review, Vol. 42 (1), pp. 43-67, March 2000. 
% 	[2] Blu, T. & Unser, M.,"The Fractional Spline Wavelet Transform: Definition and Implementation", Proc. IEEE International 
%       Conference on Acoustics, Speech, and Signal Processing (ICASSP'00), Istanbul, Turkey, Vol. {I}, pp. 512-515, June 5--9, 2000.  
%   [3] Blu, T. & Unser, M.,"A Complete Family of Scaling Functions: The $($-Fractional Splines", Proc. IEEE International 
%       Conference on Acoustics, Speech, and Signal Processing (ICASSP'03), Hong Kong, China, Vol. {VI}, pp. 421-424, April 6--10, 2003. 

M=length(w);
if M~=2^round(log(M)/log(2))
	disp(' ')
	disp('The size of the input signal must be a power of two!')
	disp(' ')
	x=[];
	return
end
if length(FFTsynthesisfilters)~=M
	disp(' ')
	disp('The size of the input signal and of the filters must match!')
	disp(' ')
	w=[];
	return
end

%
% Reconstruction of the signal from its
% bandpass components
%

G=conj(FFTsynthesisfilters(1,:));
H=conj(FFTsynthesisfilters(2,:));

M=M/2^J;
y=w(length(w)+((-M+1):0));
w=w(1:(length(w)-M));
Y=fft(y,M);
for j=J:-1:1
	z=w(length(w)+((-M+1):0));
	w=w(1:(length(w)-M));
	Z=fft(z,M);
	M=2*M;
	
	H1=H(1:2^(j-1):length(H));
	G1=G(1:2^(j-1):length(G));
	
	Y0=G1(1:M/2).*Y+H1(1:M/2).*Z;
	Y1=G1(M/2+(1:M/2)).*Y+H1(M/2+(1:M/2)).*Z;
	Y=[Y0 Y1];
end

x=real(ifft(Y,M));

