function w=FFTwaveletanalysis2D(im,FFTanalysisfilters,J);
% 	Usage: w=FFTwaveletanalysis2D(im,FFTanalysisfilters,J)
%
%   FFT-based implementation of the wavelet transform (see ref. [2]). It computes the wavelet 
%   transform of an image "im" using a Fourier method. It uses periodic boundary conditions.  The image 
%   must be square and its size be a power of two; the frequency responses (same length as the size 
%   of the image) of the analysis filters are specified in the two rows of FFTanalysisfilters.
% 	
% 	Input:
% 	im = input image, of size 2^Nx2^N=length of FFTanalysisfilters
% 	J  = depth of the decomposition, i.e., 3J+1 wavelet bands in two dimensions 
% 	FFTanalysisfilters = [lowpassfilter;highpassfilter]
% 	
% 	Output:
% 	w = image of the same size as im in the usual representation; e.g., after one iteration
%       w is made of four quadrants [LowLow LowHigh ; HighLow HighHigh], the top left being 
%       the low-pass filtered image on which further iterations may be applied.
% 	
% 	See also FFTWAVELETSYNTHESIS2D, FFTFRACTSPLINEFILTERS, WEXTRACT2D.
% 	
% 	Author: Manuela Feilner and Thierry Blu, October 1999
% 	Biomedical Imaging Group, EPFL, Lausanne, Switzerland.
% 	
% 	This software is downloadable at http://www.ee.cuhk.edu.hk/~tblu/demos/
% 	
% 	References:
% 	[1] Unser, M. & Blu, T.,"Fractional Splines and Wavelets", SIAM Review, Vol. 42 (1), pp. 43-67, March 2000. 
% 	[2] Blu, T. & Unser, M.,"The Fractional Spline Wavelet Transform: Definition and Implementation", Proc. IEEE International 
%       Conference on Acoustics, Speech, and Signal Processing (ICASSP'00), Istanbul, Turkey, Vol. {I}, pp. 512-515, June 5--9, 2000.  
%   [3] Blu, T. & Unser, M.,"A Complete Family of Scaling Functions: The $($-Fractional Splines", Proc. IEEE International 
%       Conference on Acoustics, Speech, and Signal Processing (ICASSP'03), Hong Kong, China, Vol. {VI}, pp. 421-424, April 6--10, 2003. 
%
%	Demo at http://bigwww.epfl.ch/demo/jfractsplinewavelet/

[M,MM]=size(im);
if M~=2^round(log(M)/log(2))
	disp(' ')
	disp('The size of the input signal must be a power of two!')
	disp(' ')
	w=[];
	return
end

if M~=MM
	disp(' ')
	disp('The image must be square!')
	disp(' ')
	w=[];
	return
end

if length(FFTanalysisfilters)~=M
	disp(' ')
	disp('The size of the input signal and of the filters must match!')
	disp(' ')
	w=[];
	return
end


G=FFTanalysisfilters(1,:);
H=FFTanalysisfilters(2,:);


w=zeros(M,M);
for j=1:J,
	%
	% Computation of the outputs y and z
	%
	for i=1:M,
		X=fft(im(i,:),M);
		Y=G.*X;
		Z=H.*X;
		Y=1/2*(Y(1:M/2)+Y(M/2+(1:M/2)));
		Z=1/2*(Z(1:M/2)+Z(M/2+(1:M/2)));
		z=ifft(Z,M/2);
		im(i,:)=[ifft(Y,M/2) z];
	end;
	for i=1:M,
		X=fft(im(:,i).');
		Y=G.*X;
		Z=H.*X;
		Y=1/2*(Y(1:M/2)+Y(M/2+(1:M/2)));
		Z=1/2*(Z(1:M/2)+Z(M/2+(1:M/2)));
		z=ifft(Z,M/2);
		im(:,i)=[ifft(Y,M/2) z].';		
	end;
	w(1:M,1:M)=im;

	M=M/2;
	im=w(1:M,1:M);
	G=G(1:2:length(G));
	H=H(1:2:length(H));
end;
w=real(w);