function [UR,UO]=parafit_interferogram(params)
%This program implements the single-frame complex wave retrieval algorithm in the paper 
% Luo G., He Y., Shu X., Zhou R., and Blu T. "Complex Wave and Phase Retrieval from A 
% Single Off-Axis Interferogram", accepted by JOSA A (2022)
% Acquisition model: input = |UO + UR|^2 where UR = AR.*exp(j*w^Tr) in off-axis setting;
% FUNCTION : retireve the object wave from a given real interferogram by parametric fit.
% -----------  
% INPUT
%      params.input       -- interferogram;
%      params.w           -- 1x2, reference wave frequency (wx,wy);
%      params.VR          -- pure reference wave sinusoid, VR = exp(j*w^Tr);
%      params.rho         -- circular-shaped bandwidth radius of the object wave, e.g. sqrt(2*pi)/2
%      params.epsilon     -- circular- or rectangular shaped bandwidth radius of the reference wave amplitude
%      params.iterMax     -- maximum iteration number, e.g.10
% OUTPUT:
%      UR      -- estimated reference wave
%      UO      -- estimated object wave
%
% REFERENCE:
%     [1] G. Luo, R. Guo, R. Zhou, and T. Blu, "Iterative Single-frame 
%         High-bandwidth Complex Wave Retrieval Algorithm." In Digital 
%         Holography and Three-Dimensional Imaging, pp. M5A-5. 
%         Optica Publishing Group, 2022.
%     [2] G. Luo, Y. He, X. Shu, R. Zhou, and T. Blu, "Complex Wave and
%         Phase Retrieval from A Single Off-Axis Interferogram". JOSA A. Accepted (2022).
%
% AUTHOR : Gang Luo and Thierry Blu
% DATE   : November 2022
% CONTACT: Gang Luo (luogang@link.cuhk.edu.hk), The Chinese University of Hong Kong.
%
%
% EXAMPLE OF USE:
%%%%exact reconstruction
% % clear;clc;close all;
% % addpath('Utilities/');
% % addpath('Functions/');
% % addpath('images/');
% % M=512;N=512;        
% % x=repmat((1:M)',1,N);x=(x-(M+1)/2);
% % y=repmat(1:N,M,1);y=(y-(N+1)/2);
% % I = double(imread('lena.jpg'));UO=I/max(abs(I(:)));
% % I = double(imread('boat.png'));UO=UO.*exp(2*pi*i*I/max(I(:)));
% % fx=60.2745312/M;fy=65.96785214/N;       
% % params.w=[2*pi*fx,2*pi*fy];
% % params.VR=exp(i*params.w(1)*x+i*params.w(2)*y);
% % UR=(1+randn(M,N)).*params.VR;
% % params.rho=(M/(2+3*sqrt(2))-1).*2*pi/M;
% % params.epsilon=1.1*sqrt((pi/M)^2+(pi/N)^2);
% % NO=(3*params.rho/norm(params.w,2)-1)/2
% % wrap=@(w)-mod(-w+pi,2*pi)+pi;
% % filtUO=@(I)imagefilter(I.*conj(params.VR),@(wx,wy)double(wrap(wx+params.w(1)).^2+wrap(wy+params.w(2)).^2<=params.rho^2),1).*params.VR;
% % filtAR=@(I)imagefilter(I.*conj(params.VR),@(wx,wy)double(wx.^2+wy.^2<=params.epsilon^2),1).*params.VR;
% % c=@(I)255*I/max(abs(I(:)));
% % I0=randn(size(I));I0(1,1)=1;
% % UO=filtUO(UO);
% % UR=filtAR(UR);UR=1*UR/mean(abs(UR(:)));
% % max(max(abs(UR-filtAR(UO+UR))))
% % max(max(abs(UO-filtUO(UO+UR))))
% % 
% % params.input=abs(UO+UR).^2;  
% % params.iterMax = 100;
% % [ARest,UOest]=parafit_interferogram(params);
% % figure,imview(imcontrast( abs(UO),abs(UOest) ))
% % figure,imview(imcontrast( angle(UO),angle(UOest) ))
% % PSNR(abs(UOest),abs(UO))
% % PSNR(UOest*exp(i*angle(UOest(:)\UO(:))),UO)
%

[M,N] = size(params.input);
input = params.input;

warning off;
if ~isfield(params,'rho')%check if the struct contains varible of size inside
    error('Please set the circular-shaped bandwidth radius of the object wave');
end

%% default parameters
if ~isfield(params,'iterMax')
    params.iterMax = 10;
end
if ~isfield(params,'epsilon')
    params.epsilon = 2*sqrt((pi/M)^2+(pi/N)^2);
end
if ~isfield(params,'w')
    x=repmat((1:M)',1,N);x=(x-(M+1)/2);
    y=repmat(1:N,M,1);y=(y-(N+1)/2);
    Pupil_DC_remove = x.^2 + y.^2 >=10^2;
    [wx,wy] = aux_fre_reference(input,Pupil_DC_remove);
    params.w(1) = wx;
    params.w(2) = wy;
end
if ~isfield(params,'VR')
    x=repmat((1:M)',1,N);x=(x-(M+1)/2);
    y=repmat(1:N,M,1);y=(y-(N+1)/2);
    params.VR = exp(1j*params.w(1).*x + 1j*params.w(2).*y);
end
%% bandlimiataion functions
w=params.w;
rho=params.rho;
epsilon=params.epsilon;
iterMax = params.iterMax;
VR = params.VR;

wrap=@(w)-mod(-w+pi,2*pi)+pi;
filtUO=@(I)imagefilter(I.*conj(VR),@(wx,wy)double(wrap(wx+w(1)).^2+wrap(wy+w(2)).^2<=rho^2&wx.^2+wy.^2>epsilon^2),1).*VR;%object wave bandlimiation
filtAR=@(I)real( imagefilter(I.*conj(VR),@(wx,wy)double(wx.^2+wy.^2<=epsilon^2),1) ).*VR;%reference wave bandlimiation
% filtAR=@(I)real( imagefilter(I.*conj(VR),@(wx,wy)double(abs(wx)<=epsilon & abs(wy)<=epsilon),1) ).*VR;%reference wave bandlimiation

%% apply complex wave retrieval
%initialization
phi=angle(VR);

%   stops under either a fixed iteration number or small changes in two adjacent iterations
criterion = false(1);
sqrt_I = sqrt(input);
for j=1:iterMax
    U=exp(1i*phi).*sqrt_I;
    UR=filtAR(U);
    UO=filtUO(U);
    phi=angle(UR+UO);    
    if criterion
        y_psnr_now = PSNR( abs(UR+UO),sqrt_I );
        if j>1 && y_psnr_now - y_psnr_last<0.1
            break;
        end
        y_psnr_last = y_psnr_now;
    end
%     disp(['PSNR = ' num2str(PSNR(abs(UR+UO),sqrt_I)) ' dB'])
end
end


function [wx,wy] = aux_fre_reference(interferogram,Pupil_DC_remove)
% this program is to find the reference wave frequency by locating the
% the peak positions of the twin images in the Fourier domain of the
% interferogram.
%
% return spatial frequnecy (wx,wy) where -pi<wx<=pi, -pi<wy<=pi
    [M,N] = size(interferogram);
    fft_img = fftshift(fft2(interferogram,M,N)).*Pupil_DC_remove;
    abs_fft_img = abs(fft_img);
    [row_index,col_index] = find( abs_fft_img == max(max(abs_fft_img(:,N/2:N)) ));
    wx = 2*pi/M*( -(M/2+1) + row_index(2) );
    wy = 2*pi/N*(-(N/2+1) + col_index(2) ); 
end


