/***************************************************************/
/*                                                             */
/*   LC-3b Assembler                                           */
/*                                                             */
/*   CEG3420 Lab2                                              */
/*   The Chinese University of Hong Kong                       */
/*                                                             */
/***************************************************************/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <stddef.h>

#define MAX_LABEL_LENGTH 20
#define MAX_LABELS 		 255

/* return values for isLabel function */
enum
{
	FIRST_CHAR_ERROR, OTHER_CHAR_ERROR, VALID_LABEL
};

/* errLine is the line num of the place occurs this error */
void errPrint ( int errCode, int errLine );

/* return the Reg Number if reg is legal
 * lineNum is for errPrint function */
int regNum ( char *reg, int lineNum );

/* return the pStr's value if it's in format of #... or x... */
int toNum ( char *pStr, int lineNum );

/* return the imm5 value if imm5Num is legal
 * lineNum is for errPrint function */
int imm5Val ( char *imm5Num, int lineNum );

/* return the amount4 value if amount4Num is legal
 * lineNum is for errPrint function */
int amount4Val ( char *amount4Num, int lineNum );

/* return the trapvect8 value if trap8Num is legal
 * lineNum is for errPrint function */
int trap8Val ( char *trap8Num, int lineNum );

/* struct to store label table
 * label saves the label in original Assembler program
 * address saves the line number of labels
 * table end with a label == '\0' */

typedef struct label_table {
	int lineCnt;
	int address;
	char label[MAX_LABEL_LENGTH+1];
} TableEntry;

/* test if lLabel is a legal label name*/
int isLabel(char *lLabel/*, TableEntry *labelTable*/);

/* return the value of LABEL segment if label if legal
 * labelWid is the number of bits for representing this label
 * lineNum is for LABEL Value calculation & errPrint function */
int labelVal ( char *labelName, int labelWid, TableEntry *pLabel, int cmdNum, int lineNum );

/* return the unsigned value of a constant used in cmdParse for Code Joint
 * For example, toUnsig(-5, 4) = 11, because -5 = 'b1011, trans it to
 * unsigned number is 11. Note that toUnsig(posNum, width) = posNum
 * width <= 12 */
int toUnsig( int num, int width );

/* Joint opCode and all arguments to a complete machine code */
int codeJoint (int opCode, int arg1, int arg1Wid, int arg2, int arg2Wid,
               int arg3, int arg3Wid, int arg4, int arg4Wid);

/* Operand Number Check, if the number of arguments is different from opNum, the program will exit */
void opNumChk ( int opNum, char *pArg1, char *pArg2, char *pArg3, char *pArg4, int lineNum );
			   
/* test if reg is in the format of a register  */
int isReg ( char *reg );

/* test if reg is in the format of a immediate Number */
int isImm ( char *imm );

/* return the translated assembly code (machine code) if cmd is legal
 * lineNum is the line Number of the cmd in source file
 * cmdNum is the ordinal of the cmd in source file */
int cmdParser ( int lineNum, int cmdNum, char *pOpcode, char *pArg1, char *pArg2,
				char *pArg3, char *pArg4, TableEntry *pLabel );

/************************* cmdParser *************************/

#define MAX_LINE_LENGTH 255

enum
{
   DONE, OK, EMPTY_LINE
};

enum
{
	ADD, AND, BR, HALT, JMP, JSR, JSRR, LDB, LDW, LEA, NOP, NOT, RET, LSHF, RSHFL, RSHFA, RTI, STB, STW, TRAP, XOR, NOTMATCH
};

int isOpcode(char *lPtr);
int readAndParse( FILE * pInfile, char * pLine, char ** pLabel, char ** pOpcode, char ** pArg1, char ** pArg2, char ** pArg3, char ** pArg4);
void cmdProcess(char *infileName, char *outfileName);
void throwError(int exitCode, const char *format, ...);

/************************* cmdProcess *************************/

int main(int argc, char *argv[])
{
	if (argc < 2)
	{
		throwError(4, "%s: error: lack of input file or output file", argv[0]);
	}
	cmdProcess(argv[1], argv[2]);

	return 0;
}

/*************************************************************/

void errPrint ( int errCode, int errLine )
{
	switch ( errCode )
	{
		case 1: 
			printf ( "Error 1: Undefined label in line %d.\n", errLine+1 );
			printf ("exit code: %d\n", 1);
			exit(1);
			break;
		case 2:
			printf ( "Error 2: Invalid opcode in line %d.\n", errLine+1 );
			printf ("exit code: %d\n", 2);
			exit(2);
			break;
		case 3:
			printf ( "Error 3: Invalid constant in line %d.\n", errLine+1 );
			printf ("exit code: %d\n", 3);
			exit(3);
			break;
		case 4:
			printf ( "Error 4: Invalid Reg operand in line %d.\n", errLine+1 ); 
			printf ("exit code: %d\n", 4);
			exit(4);
			break;
		case 5:
			printf ( "Error 5: Wrong Number of  operands in line %d.\n", errLine+1 ); 
			printf ("exit code: %d\n", 4);
			exit(4);
			break;
		case 6:
			printf ( "Error 6: Error occurs in line %d.\n", errLine+1 ) ;
			printf ("exit code: %d\n", 4);
			exit(4);
			break;
		case 7:
			printf ( "Error 7: Unexpected operand in line %d.\n", errLine+1 ) ;
			printf ("exit code: %d\n", 4);
			exit(4);
			break;
		case 8:
			printf ( "Error 8: Label in line %d is too far to fetch.\n", errLine+1 ) ;
			printf ("exit code: %d\n", 4);
			exit(4);
			break;
		case 9:
			printf ( "Error 9: Invalid memory address to load program in line %d.\n", errLine+1 );
			printf ("exit code: %d\n", 3);
			exit(3);
			break;
		case 10:
			printf ( "Error 10: Invalid label name in line %d.\n", errLine+1 );
			printf ("exit code: %d\n", 4);
			exit(4);
			break;
		/* reserved for other Error Code */

		case 99:
			printf ( "Error 99: An error occured in line %d.\n", errLine+1 ) ;
			printf ("exit code: %d\n", 4);
			exit(4);
			break;
		default:
			printf ( "Opps! The errPrint function has BUGS!\n" ) ;
			exit(4);
			break;
	}
}

int toNum ( char *pStr, int lineNum )
{
	/* neg == 0: positive number  neg == 1: negative number */
	int i, neg = 0;
	if ( *pStr == '#' )
	{
		pStr++;
		if ( *pStr == '-' )
		{
			neg = 1;
			pStr++;
		}
		int len = strlen(pStr);
		char *tPtr = pStr;
		for ( i = 0; i < len; i++ )
		{
			if (!isdigit(*tPtr) )
			{
				errPrint(3, lineNum);
			}
			tPtr++;
		}
		int num = atoi(pStr);
		if ( neg )
		{
			num = -num;
		}
		return num;
	}
	else if ( *pStr == 'x' )
	{
		pStr++;
		if ( *pStr == '-' )
		{
			neg = 1;
			pStr++;
		}
		int len = strlen(pStr);
		char *tPtr = pStr;
		for ( i = 0; i < len; i++ )
		{
			if (!isxdigit(*tPtr) )
			{
				errPrint(3, lineNum);
			}
			tPtr++;
		}
		int num = strtol(pStr, NULL, 16);
		if ( neg )
		{
			num = -num;
		}
		return num;
	}
	else
	{
		errPrint(3, lineNum);
	}	
    return 0;
}

int isReg ( char *reg )
{
	if ( reg[0] == 'r' && isdigit(reg[1]) )
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

int regNum ( char *reg, int lineNum )
{
	if ( !isReg(reg) )
	{
		errPrint (7, lineNum);
        return 0;
	}
	else if ( strlen(reg) != 2 || atoi(&reg[1]) > 7 )
	{
		errPrint(4, lineNum);
        return 0;
	}
	else
	{
		return atoi(&reg[1]);
	}
}

int isImm ( char *immNum )
{
	if (immNum[0] == '#' || immNum[0] == 'x')
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

int amount4Val ( char *amount4Num, int lineNum ) 
{	
	int amount4;
	if (!isImm(amount4Num))
	{
		errPrint (7, lineNum);
	}
	else
	{
		amount4 = toNum(amount4Num, lineNum);
		if ( amount4 > 15 || amount4 < 0 )
		{
			errPrint(3, lineNum);
		}
	}
	return amount4;
}

int imm5Val ( char *imm5Num, int lineNum )
{
	int imm5;
	if (!isImm(imm5Num)) {
		errPrint (7, lineNum);
	}
	else
	{
		imm5 = toNum(imm5Num, lineNum);
		if ( imm5 > 15 || imm5 < -16 )
		{
			errPrint(3, lineNum);
		}			
	}
	return imm5;
}

int boffset6Val ( char *boffset6Num, int lineNum )
{
	int boffset6;
	if (!isImm(boffset6Num))
	{
		errPrint (7, lineNum);
	}
	else
	{
		boffset6 = toNum(boffset6Num, lineNum);
		if ( boffset6 > 31 || boffset6 < -32 )
		{
			errPrint(3, lineNum);
		}			
	}
	return boffset6;
}

int trap8Val ( char *trap8Num, int lineNum )
{
	/* int trap8; */
	if ( trap8Num[0] == '#' )
	{
		errPrint(3, lineNum);
	}
	else if (trap8Num[0] != 'x')
	{
		errPrint(7, lineNum);
	}

   	int trap8 = toNum(trap8Num, lineNum);

	if ( trap8 > 255 || trap8 < 0 )
	{
		errPrint(3, lineNum);
	}
	return trap8;
}

int isLabel(char *lLabel/*, TableEntry *labelTable*/)
{
	char *pLabel = lLabel;
	int i;
	if (*pLabel == 'x' || isdigit(*pLabel)) return FIRST_CHAR_ERROR;
	else
	{
		pLabel++;
		while (pLabel != NULL && *pLabel != '\0')
		{
			/* all characters have been converted to lower-case */
			if (isalnum(*pLabel)) {pLabel++; continue;}
			else return OTHER_CHAR_ERROR;
		}
	}

	return VALID_LABEL;
}

int labelVal ( char *labelName, int labelWid, TableEntry *pLabel, int cmdNum, int lineNum )
{
	if (isLabel(labelName) != VALID_LABEL)
	{
		errPrint(10, lineNum);
	}
	TableEntry *tPtr = pLabel;
	while ( *(tPtr->label) != '\0' )
	{
		if ( strcmp(tPtr-> label, labelName) == 0)
		{
			int offset = ((tPtr->address) - cmdNum - 1);
			if (offset >= (1<<labelWid) || offset < -(1<<labelWid))
			{
				errPrint(8, lineNum);
			}
			return offset;
		}
		tPtr++;
	}
	errPrint(1, lineNum);
    return 0;
}

int toUnsig ( int num, int width )
{
		int seg = num & 0x0FFF;
		int mod = (1<<width);
		return seg % mod;
}

void opNumChk ( int opNum, char *pArg1, char *pArg2, char *pArg3, char *pArg4, int lineNum )
{
	switch ( opNum )
	{
		case 0:
			if (!( pArg1 == NULL && pArg2 == NULL && pArg3 == NULL && pArg4 == NULL ))
				errPrint(5, lineNum);
			break;
		case 1:
			if (!( pArg1 != NULL && pArg2 == NULL && pArg3 == NULL && pArg4 == NULL ))
				errPrint(5, lineNum);
			break;
		case 2: 
			if (!( pArg1 != NULL && pArg2 != NULL && pArg3 == NULL && pArg4 == NULL ))
				errPrint(5, lineNum);
			break;
		case 3: 
			if (!( pArg1 != NULL && pArg2 != NULL && pArg3 != NULL && pArg4 == NULL ))
			errPrint(5, lineNum);
				break;
		case 4: 
		if (!( pArg1 != NULL && pArg2 != NULL && pArg3 != NULL && pArg4 != NULL ))
			errPrint(5, lineNum);
			break;
		default: 
			printf ("Illegal use of 'opNumChk' Function!\n");
	}
}

int codeJoint (int opCode, int arg1, int arg1Wid, int arg2, int arg2Wid,
               int arg3, int arg3Wid, int arg4, int arg4Wid)
{
	int machCode = opCode;
	machCode = ( machCode << arg1Wid )+ arg1;
	machCode = ( machCode << arg2Wid )+ arg2;
	machCode = ( machCode << arg3Wid )+ arg3;
	machCode = ( machCode << arg4Wid )+ arg4;
	return machCode;
}

int cmdParser( int lineNum, int cmdNum, char *pOpcode, char *pArg1, char *pArg2,
			   char *pArg3, char *pArg4, TableEntry *pLabel )
{	
	int machCode = 0;  /* For save the translated Machine Code */
	/* NOP Instruction */
	if ( !strcmp(pOpcode, "nop") )
	{
		opNumChk(0, pArg1, pArg2, pArg3, pArg4, lineNum);
		return 0;
	}
	/* .ORIG */
	else if ( !strcmp(pOpcode, ".orig") )
	{
		opNumChk(1, pArg1, pArg2, pArg3, pArg4, lineNum);
		int startAddr = toNum(pArg1, lineNum);
		/* if ( startAddr > 0xFDFF || startAddr < 0x3000 || startAddr % 2 ) */
		if ( startAddr >  0xFFFF || startAddr < 0x0000 || startAddr % 2 )
		{
			errPrint(9, lineNum);
		}
		return startAddr;
	}
	/* .FILL */
	else if ( !strcmp(pOpcode, ".fill") )
	{
		opNumChk(1, pArg1, pArg2, pArg3, pArg4, lineNum);
		int fillAddr = toNum(pArg1, lineNum);
		if ( fillAddr > 0x7FFF || fillAddr < -(0x8000) )
		{
			errPrint(3, lineNum);
		}
		return fillAddr;
	}
	/* ADD : 0001 */
	else if ( !strcmp(pOpcode, "add") )
	{
		opNumChk(3, pArg1, pArg2, pArg3, pArg4, lineNum);
		if ( isReg(pArg3) )  /* ADD DR, SR1, SR2 */
		{
			return codeJoint ( 1,
							   regNum(pArg1, lineNum), 3,
							   regNum(pArg2, lineNum), 3,
							   0, 3,
							   regNum(pArg3, lineNum), 3);
		}
		else  /* ADD DR, SR1, imm5 */
		{
			return codeJoint ( 1,
							   regNum(pArg1, lineNum), 3,
							   regNum(pArg2, lineNum), 3,
							   1, 1,
							   toUnsig(imm5Val(pArg3, lineNum), 5), 5);
		}
	}	
	/* AND : 0101 */
	else if ( !strcmp(pOpcode, "and") )
	{
		opNumChk(3, pArg1, pArg2, pArg3, pArg4, lineNum);
		if ( isReg(pArg3) )  /* AND DR, SR1, SR2 */
		{  
			return codeJoint ( 5,
							   regNum(pArg1, lineNum), 3,
							   regNum(pArg2, lineNum), 3,
							   0, 3,
							   regNum(pArg3, lineNum), 3);
		}
		else  /* AND DR, SR1, imm5 */
		{
			return codeJoint ( 5,
							   regNum(pArg1, lineNum), 3,
							   regNum(pArg2, lineNum), 3,
							   1, 1,
							   toUnsig(imm5Val(pArg3, lineNum), 5), 5);
		}
	}	
	/* BR : 0000 */
	else if ( !strcmp(pOpcode, "br" )  ||
			  !strcmp(pOpcode, "brn")  ||
			  !strcmp(pOpcode, "brz")  ||
			  !strcmp(pOpcode, "brp")  ||
			  !strcmp(pOpcode, "brnz") ||
			  !strcmp(pOpcode, "brnp") ||
			  !strcmp(pOpcode, "brzp") ||
			  !strcmp(pOpcode, "brnzp")) 
	{
		int pcoffset9, condition = 0;
		opNumChk(1, pArg1, pArg2, pArg3, pArg4, lineNum);
		if ( strchr(pOpcode, 'n') != NULL ) condition = condition + 4;
		if ( strchr(pOpcode, 'z') != NULL ) condition = condition + 2;
		if ( strchr(pOpcode, 'p') != NULL ) condition = condition + 1;
		condition = (condition == 0) ? 7 : condition; /* br is interpreted the same as brnzp */
		pcoffset9 = labelVal(pArg1, 9, pLabel, cmdNum, lineNum);
		return codeJoint (0,
					      condition, 3,
						  toUnsig(pcoffset9, 9), 9,
						  0, 0,
						  0, 0);
	}	
	/* JMP : 1100 */
	else if ( !strcmp(pOpcode, "jmp") )
	{
		opNumChk(1, pArg1, pArg2, pArg3, pArg4, lineNum);
		return codeJoint ( 12,
						   0, 3,
						   regNum(pArg1, lineNum), 3,
						   0, 6,
						   0, 0);
	}
	/* RET : 1100 */
	else if ( !strcmp(pOpcode, "ret") )
	{
		opNumChk(0, pArg1, pArg2, pArg3, pArg4, lineNum);
		return codeJoint ( 12,
						   0, 3,
						   7, 3,
						   0, 6,
						   0, 0);
	}
	/* JSR : 0100
	 * JSRR: 0100 */
	else if ( !strcmp(pOpcode, "jsr") || !strcmp(pOpcode, "jsrr") )
	{
		opNumChk(1, pArg1, pArg2, pArg3, pArg4, lineNum);
		if ( isReg(pArg1) ) {      /* JSRR BaseR */
			return codeJoint ( 4,
							   0, 3,
							   regNum(pArg1, lineNum), 3,
							   0, 6,
							   0, 0);
		}
		else
		{  /* JSR LABEL */
			return codeJoint ( 4,
							   1, 1,
							   toUnsig(labelVal(pArg1, 11, pLabel, cmdNum, lineNum), 11), 11,
							   0, 0,
							   0, 0);
		}
	}
	/* LDB : 0010 */
	else if ( !strcmp(pOpcode, "ldb") )
	{
		opNumChk(3, pArg1, pArg2, pArg3, pArg4, lineNum);
		return codeJoint ( 2,
						   regNum(pArg1, lineNum), 3,
						   regNum(pArg2, lineNum), 3,
						   toUnsig(boffset6Val(pArg3, lineNum), 6), 6,
						   0, 0);
	}
	/* LDW : 0110 */
	else if ( !strcmp(pOpcode, "ldw") )
	{
		opNumChk(3, pArg1, pArg2, pArg3, pArg4, lineNum);
		return codeJoint ( 6,
						   regNum(pArg1, lineNum), 3,
						   regNum(pArg2, lineNum), 3,
						   toUnsig(boffset6Val(pArg3, lineNum), 6), 6,
						   0, 0);
	}
	/* LEA : 1110 */
	else if ( !strcmp(pOpcode, "lea") )
	{
		opNumChk(2, pArg1, pArg2, pArg3, pArg4, lineNum);
		return codeJoint ( 14,
						   regNum(pArg1, lineNum), 3,
						   toUnsig(labelVal(pArg2, 9, pLabel, cmdNum, lineNum), 9), 9,
						   0, 0,
						   0, 0);
	}
	/* NOT : 1001 */
	else if ( !strcmp(pOpcode, "not") )
	{
		opNumChk(2, pArg1, pArg2, pArg3, pArg4, lineNum);
		return codeJoint ( 9,
						   regNum(pArg1, lineNum), 3,
						   regNum(pArg2, lineNum), 3,
						   63, 6,
						   0, 0);
	}
	/* XOR : 1001 */
	else if ( !strcmp(pOpcode, "xor") )
	{
		opNumChk(3, pArg1, pArg2, pArg3, pArg4, lineNum);
		if ( isReg(pArg3) )  /* XOR DR, SR1, SR2 */
		{
			return codeJoint ( 9,
							   regNum(pArg1, lineNum), 3,
							   regNum(pArg2, lineNum), 3,
							   0, 3,
							   regNum(pArg3, lineNum), 3);
		}
		else  /* XOR DR, SR, imm5 */
		{
			return codeJoint ( 9,
							   regNum(pArg1, lineNum), 3,
							   regNum(pArg2, lineNum), 3,
							   1, 1,
							   toUnsig(imm5Val(pArg3, lineNum), 5), 5);
		}
	}
	/* RTI : 1000 */
	else if ( !strcmp(pOpcode, "rti") )
	{
		opNumChk(0, pArg1, pArg2, pArg3, pArg4, lineNum);
		return codeJoint ( 8,
						   0, 12,
						   0, 0,
						   0, 0,
						   0, 0);
	}
	/* LSHF : 1101
	 * RSHFL: 1101
	 * RSHFA: 1101 */
	else if ( !strcmp(pOpcode, "lshf")  ||
			  !strcmp(pOpcode, "rshfl") ||
			  !strcmp(pOpcode, "rshfa") )
	{
		opNumChk(3, pArg1, pArg2, pArg3, pArg4, lineNum);
		int condition = 0;
		if ( pOpcode[0] == 'r' )
		{
			condition++;
			if (pOpcode[4] == 'a')
			{
				condition = condition + 2;
			}
		}
		return codeJoint ( 13,
						   regNum(pArg1, lineNum), 3,
						   regNum(pArg2, lineNum), 3,
						   condition, 2,
						   amount4Val(pArg3, lineNum), 4);		
	}
	/* STB : 0011 */
	else if ( !strcmp(pOpcode, "stb") )
	{
		opNumChk(3, pArg1, pArg2, pArg3, pArg4, lineNum);
		return codeJoint ( 3,
						   regNum(pArg1, lineNum), 3,
						   regNum(pArg2, lineNum), 3,
						   toUnsig(boffset6Val(pArg3, lineNum), 6), 6,
						   0, 0);
	}
	/* STW : 0111 */
	else if ( !strcmp(pOpcode, "stw") )
	{
		opNumChk(3, pArg1, pArg2, pArg3, pArg4, lineNum);
		return codeJoint ( 7,
						   regNum(pArg1, lineNum), 3,
						   regNum(pArg2, lineNum), 3,
						   toUnsig(boffset6Val(pArg3, lineNum), 6), 6,
						   0, 0);
	}
	/* TRAP : 1111 */
	else if ( !strcmp(pOpcode, "trap" ) )
	{
		opNumChk(1, pArg1, pArg2, pArg3, pArg4, lineNum);
		return codeJoint ( 15,
						   0, 4,
						   trap8Val(pArg1, lineNum), 8,
						   0, 0,
						   0, 0);
	}
	/* HALT : TRAP x25 */
	else if ( !strcmp(pOpcode, "halt") )
	{
		return codeJoint ( 15,
						   0, 4,
						   0x25, 8,
						   0, 0,
						   0, 0);
	}
	else
	{
		errPrint(2, lineNum);
        return 0;
	}
}

/*************************************************************/

int isOpcode(char *lPtr)
{
	if (strcmp(lPtr, "add") == 0) return ADD;
	else if (strcmp(lPtr, "and") == 0) return AND;
	else if (strncmp(lPtr, "br", 2) == 0) return BR;
	else if (strcmp(lPtr, "halt") == 0) return HALT;
	else if (strcmp(lPtr, "jmp") == 0) return JMP;
	else if (strcmp(lPtr, "jsr") == 0) return JSR;
	else if (strcmp(lPtr, "jsrr") == 0) return JSRR;
	else if (strcmp(lPtr, "ldb") == 0) return LDB;
	else if (strcmp(lPtr, "ldw") == 0) return LDW;
	else if (strcmp(lPtr, "lea") == 0) return LEA;
	else if (strcmp(lPtr, "nop") == 0) return NOP;
	else if (strcmp(lPtr, "not") == 0) return NOT;
	else if (strcmp(lPtr, "ret") == 0) return RET;
	else if (strcmp(lPtr, "lshf") == 0) return LSHF;
	else if (strcmp(lPtr, "rshfl") == 0) return RSHFL;
	else if (strcmp(lPtr, "rshfa") == 0) return RSHFA;
	else if (strcmp(lPtr, "rti") == 0) return RTI;
	else if (strcmp(lPtr, "stb") == 0) return STB;
	else if (strcmp(lPtr, "stw") == 0) return STW;
	else if (strcmp(lPtr, "trap") == 0) return TRAP;
	else if (strcmp(lPtr, "xor") == 0) return XOR;
	else return NOTMATCH;
}

int readAndParse( FILE * pInfile, char * pLine, char ** pLabel, char ** pOpcode, char ** pArg1, char ** pArg2, char ** pArg3, char ** pArg4)
{
	char *lPtr;
	int i;
	if( !fgets( pLine, MAX_LINE_LENGTH, pInfile ) )
		return( DONE );

	/* convert entire line to lowercase */
	for( i = 0; i < strlen( pLine ); i++ )
		pLine[i] = tolower( pLine[i] );

	/* make the pointers point to the address of a '\0' */
	*pLabel = *pOpcode = *pArg1 = *pArg2 = *pArg3 = *pArg4 = pLine + strlen(pLine);

	/* ignore the comments */
	lPtr = pLine;

	while( *lPtr != ';' && *lPtr != '\0' && *lPtr != '\n'  && *lPtr != '\r') 
		lPtr++;

	*lPtr = '\0';
	if( !(lPtr = strtok( pLine, "\t\n ," ) ) ) 
		return( EMPTY_LINE );

	if( isOpcode( lPtr ) == NOTMATCH && lPtr[0] != '.' ) /* found a label */
 	{
		*pLabel = lPtr;
		if( !( lPtr = strtok( NULL, "\t\n ," ) ) ) return( OK );
	}
	*pOpcode = lPtr;

	if( !( lPtr = strtok( NULL, "\t\n ," ) ) ) return( OK );
	*pArg1 = lPtr;
   
	if( !( lPtr = strtok( NULL, "\t\n ," ) ) ) return( OK );
	*pArg2 = lPtr;
	
	if( !( lPtr = strtok( NULL, "\t\n ," ) ) ) return( OK );
	*pArg3 = lPtr;

	if( !( lPtr = strtok( NULL, "\t\n ," ) ) ) return( OK );
	*pArg4 = lPtr;

	return( OK );
}

void cmdProcess(char *infileName, char *outfileName)
{
	FILE *lInfile = fopen(infileName, "r");
	FILE *lOutfile = fopen(outfileName, "w");
	char lLine[MAX_LINE_LENGTH+1];
   	char *lLabel, *lOpcode, *lArg1, *lArg2, *lArg3, *lArg4;	
	int lRet;
	int cmdCnt = 0;
	int lineCnt = 0;
	TableEntry labelTable[MAX_LABELS];
	TableEntry *pLabelEntry = labelTable;
	int i, j;
	int hasOrig = 0;

	if (lInfile == NULL)
	{
		/* printf("Cannot open file %s\n", infileName); */
		throwError(4, "Error: cannot open input file %s", infileName);
	}

	/* initialize labelTable */
	for (i = 0; i < MAX_LABELS; i++, pLabelEntry++)
		pLabelEntry->label[0] = '\0';
	pLabelEntry = labelTable;

	printf("Processing input file %s\n", infileName);
	printf("Writing result to output file %s\n", outfileName);

	do
	{
		lRet = readAndParse(lInfile, lLine, &lLabel, &lOpcode, &lArg1, &lArg2, &lArg3, &lArg4);
		if (lRet != DONE && lRet != EMPTY_LINE)
		{
			if (lLabel != NULL && lLabel[0] != '\0')
			{
				/* check if label is in a .ORIG block */
				if (!hasOrig) throwError(4, "Error: line %d: No valid .ORIG block found for this instruction", lineCnt+1);
				if ((i = isLabel(lLabel/*, labelTable*/)) == VALID_LABEL)
				{
					/* check duplicated */
					for (j = 0; j < MAX_LABELS; j++)
					{
                        if ('\0' == labelTable[j].label[0]) continue;
						if (strcmp(lLabel, labelTable[j].label) == 0)
						{
							throwError(4, "Error: line %d: current label has a conflit with another label in line %d", lineCnt+1, labelTable[j].lineCnt+1);
						}
					}
					/* add lLabel to label table */
					strncpy(pLabelEntry->label, lLabel, MAX_LABEL_LENGTH);
					pLabelEntry->label[MAX_LABEL_LENGTH] = '\0';
					pLabelEntry->address = cmdCnt;
					pLabelEntry->lineCnt = lineCnt;
					/* print value added to labelTable for debug */
					/*printf("%s, %d, %d\n", pLabelEntry->label, pLabelEntry->address, pLabelEntry->lineCnt);*/
					pLabelEntry++;
				}
				else if (i == FIRST_CHAR_ERROR)
					throwError(4, "Error: line %d: the first character of a valid label should not be \"x\" or a number", lineCnt+1);
				else if (i == OTHER_CHAR_ERROR)
					throwError(4, "Error: line %d: a valid label is supposed to be consist of alphanumeric", lineCnt+1);
			}
			cmdCnt++;

			if (strcmp(lOpcode, ".orig") == 0)
			{
				if (hasOrig) throwError(4, "Error: line %d: .ORIG block ends without .END or .ORIG appears in a .ORIG block", lineCnt+1);
				else hasOrig = 1;
			}
			else if (strcmp(lOpcode, ".end") == 0)
			{
				if (lArg1 != NULL  && lArg1[0] != '\0') throwError(4, "Error: line %d: unexpected operand", lineCnt+1);

				if (hasOrig) {hasOrig = 0; break;} /* ignore instructions after .end */
				else throwError(4, "Error: line %d: No matching .ORIG found before .END", lineCnt+1);
			}
			else
			{
				if (!hasOrig) throwError(4, "Error: line %d: No valid .ORIG block found for this instruction", lineCnt+1);
			}
		}
		lineCnt++;
	} while (lRet != DONE);

	if (lOutfile == NULL)
	{
		/* printf("Cannot open file %s\n", infileName); */
		throwError(4, "Error: cannot open output file %s", outfileName);
	}

	cmdCnt = 0;
	lineCnt = 0;
	rewind(lInfile);
	do
	{
		lRet = readAndParse(lInfile, lLine, &lLabel, &lOpcode, &lArg1, &lArg2, &lArg3, &lArg4);
		if (lRet != DONE && lRet != EMPTY_LINE)
		{
			if (strcmp(lOpcode, ".end") == 0) break;
			if (lOpcode != NULL && *lOpcode == '\0') lOpcode = NULL;
			if (lArg1 != NULL && *lArg1 == '\0') lArg1 = NULL;
			if (lArg2 != NULL && *lArg2 == '\0') lArg2 = NULL;
			if (lArg3 != NULL && *lArg3 == '\0') lArg3 = NULL;
			if (lArg4 != NULL && *lArg4 == '\0') lArg4 = NULL;
			/* check label with nothing followed */
			if (lOpcode == NULL && lArg1 == NULL && lArg2 == NULL && lArg3 == NULL && lArg4 == NULL)
				throwError(2, "Error: line %d: invalid opcode", lineCnt+1);
			fprintf(lOutfile, "0x%04X\n", cmdParser(lineCnt, cmdCnt, lOpcode, lArg1, lArg2, lArg3, lArg4, labelTable));
			cmdCnt++;
		}
		lineCnt++;
	} while (lRet != DONE);

	fclose(lInfile);
	fclose(lOutfile);
}

void throwError(int exitCode, const char *format, ...)
{
	va_list argp;

	va_start(argp, format);
	vprintf(format, argp);
	va_end(argp);

	printf("\nexit code: %d\n", exitCode);

	exit(exitCode);
}
