#include "GL4D/gl4d.h"
#include "GL4D/rolling_ball.h"

#include <math.h>
#include <stdio.h>

#ifdef __APPLE__
    #include <GLUT/glut.h>
#else
    #include <GL/glut.h>
#endif

#ifdef WIN32
#define snprintf _snprintf
#define copysign _copysign
#endif

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

#define EPSILON 1e-5

#define HYPERCUBE_SCALE 0.75
#define IDLE_REDRAW

static int g_main_window = 0;
static int g_light_4d_window = 0;

static unsigned int g_frame = 0;
static unsigned int g_timebase = 0;

static unsigned char g_high_quality = 0;

static gl4d_rb_state_t g_rb_state_3d;
static gl4d_rb_state_t g_rb_state_4d;
static void (*g_active_rb_func)(int x, int y);

static int g_light_4d_screen[2] = {0, 0};
static char g_light_control_xy = 0;
static GL4Ddouble g_light_4d_pos[4] = {0.0, 0.0, 0.0, -1.0};

static GL4Ddouble g_mesh_v[19][19][7][4];
static GL4Ddouble g_mesh_n[19][19][7][4];
static GLuint g_mesh_w = 19;
static GLuint g_mesh_h = 19;
static GLuint g_mesh_d = 7;

static GLuint g_max_display_mode = 3;
static GLuint g_display_mode = 0;
static char *g_display_mode_string[] = {
    "Immediate mode hypercube",
    "Immediate mode 4D torus",
    "Retained mode 4D torus"
};

static GLuint g_gl4d_surface = 0;

void decompose_hexahedron(GL4Ddouble v[8][4], GL4Ddouble n[8][4]) {
    GL4Duint tetrahedron_vertices[6][4] = {
        {0, 4, 2, 1},
        {4, 2, 1, 6}, /* First prism */
        {6, 3, 2, 1},

        {7, 3, 6, 5},
        {3, 6, 5, 1}, /* Second prism */
        {1, 4, 6, 5},
    };
    GL4Duint i, j;

    for (i = 0; i < 6; i++) {
        for (j = 0; j < 4; j++) {
            gl4DNormal4dv(n[tetrahedron_vertices[i][j]]);
            gl4DVertex4dv(v[tetrahedron_vertices[i][j]]);
        }
    }
}

void load_mesh(const char *mesh_file) {
    FILE *f = fopen(mesh_file, "r");
    if (f != NULL) {
        GLuint i, j, k;
        GLuint w, h, d;

        fscanf(f, "%u %u\n", &i, &j);
        fscanf(f, "%u %u %u\n", &w, &h, &d);

        for (i = 0; i < w; i++) {
            for (j = 0; j < h; j++) {
                for (k = 0; k < d; k++) {
                    char tc;
                    GLuint ti, tj, tk;

                    fscanf(f, "%c %u %u %u %lf %lf %lf %lf\n",
                        &tc, &ti, &tj, &tk,
                        &g_mesh_v[i][j][k][0],
                        &g_mesh_v[i][j][k][1],
                        &g_mesh_v[i][j][k][2],
                        &g_mesh_v[i][j][k][3]
                    );

                    fscanf(f, "%c %u %u %u %lf %lf %lf %lf\n",
                        &tc, &ti, &tj, &tk,
                        &g_mesh_n[i][j][k][0],
                        &g_mesh_n[i][j][k][1],
                        &g_mesh_n[i][j][k][2],
                        &g_mesh_n[i][j][k][3]
                    );
                }
            }
        }

        fclose(f);
    }
}

GLuint mesh_linearizer(GLuint x, GLuint y, GLuint z) {
    return x * g_mesh_h * g_mesh_d + y * g_mesh_d + z;
}

void render_bitmap_string(
    GLfloat x, GLfloat y, GLfloat z, void *font, char *string) {

    char *c;
    glRasterPos3f(x, y, z);

    for (c = string; *c; ++c) {
        glutBitmapCharacter(font, *c);
    }
}

void display_fps(void) {
    static char buffer[255] = "FPS: Calculating...";
    unsigned int time;

    g_frame++;

    time = glutGet(GLUT_ELAPSED_TIME);
    if (time - g_timebase > 1000) {
        snprintf(buffer, sizeof(buffer), "FPS: %4.2f\n",
            g_frame * 1000.0 / (time - g_timebase));

        g_timebase = time;
        g_frame = 0;
    }

    render_bitmap_string(
        -0.95f, 0.9f, 0.0f, GLUT_BITMAP_HELVETICA_12, buffer);
}

void reshape(int w, int h) {
    glViewport(0, 0, w, h);
}

void display(void) {
    GL4Ddouble v[8][4];
    GL4Ddouble n[8][4];
    GL4Dfloat ambient[4];
    GL4Dfloat diffuse[4];
    GL4Dfloat specular[4];

    glClear(GL_COLOR_BUFFER_BIT);

    {
        GL4Dfloat light[4];
        light[0] = (GL4Dfloat) g_light_4d_pos[0];
        light[1] = (GL4Dfloat) g_light_4d_pos[1];
        light[2] = (GL4Dfloat) g_light_4d_pos[2];
        light[3] = (GL4Dfloat) g_light_4d_pos[3];

        gl4DLightfv(GL4D_LIGHT0, GL4D_POSITION, light);
    }

    gl4d_rb_load_matrix(&g_rb_state_3d, gl4DLoadVolumeModelViewMatrixf);
    gl4d_rb_load_matrix(&g_rb_state_4d, gl4DLoadModelViewMatrixf);

    if (g_gl4d_surface == 0) {
        gl4DSurfacedv(
            &g_gl4d_surface,
            &g_mesh_n[0][0][0][0], &g_mesh_v[0][0][0][0],
            mesh_linearizer,
            g_mesh_w, g_mesh_h, g_mesh_d
        );
    }

    if (g_display_mode == 2) {
        gl4DActiveSurface(g_gl4d_surface);
    }

    gl4DRenderBegin();

    if (g_display_mode == 0) {
        ambient[0] = 0.75; ambient[1] = 0.10; ambient[2] = 0.10; ambient[3] = 1.0;
        diffuse[0] = 1.00; diffuse[1] = 0.25; diffuse[2] = 0.25; diffuse[3] = 1.0;
        specular[0] = 1.00; specular[1] = 1.00; specular[2] = 1.00; specular[3] = 1.0;
        gl4DMaterialfv(ambient, diffuse, specular);
        gl4DBegin(GL4D_TETRAHEDRA);
        v[0][0] =-0.50; v[0][1] =-0.50; v[0][2] =-0.50; v[0][3] =-0.50;
        n[0][0] =-1.00; n[0][1] = 0.00; n[0][2] = 0.00; n[0][3] = 0.00;
        v[1][0] =-0.50; v[1][1] =-0.50; v[1][2] =-0.50; v[1][3] = 0.50;
        n[1][0] =-1.00; n[1][1] = 0.00; n[1][2] = 0.00; n[1][3] = 0.00;
        v[2][0] =-0.50; v[2][1] =-0.50; v[2][2] = 0.50; v[2][3] =-0.50;
        n[2][0] =-1.00; n[2][1] = 0.00; n[2][2] = 0.00; n[2][3] = 0.00;
        v[3][0] =-0.50; v[3][1] =-0.50; v[3][2] = 0.50; v[3][3] = 0.50;
        n[3][0] =-1.00; n[3][1] = 0.00; n[3][2] = 0.00; n[3][3] = 0.00;
        v[4][0] =-0.50; v[4][1] = 0.50; v[4][2] =-0.50; v[4][3] =-0.50;
        n[4][0] =-1.00; n[4][1] = 0.00; n[4][2] = 0.00; n[4][3] = 0.00;
        v[5][0] =-0.50; v[5][1] = 0.50; v[5][2] =-0.50; v[5][3] = 0.50;
        n[5][0] =-1.00; n[5][1] = 0.00; n[5][2] = 0.00; n[5][3] = 0.00;
        v[6][0] =-0.50; v[6][1] = 0.50; v[6][2] = 0.50; v[6][3] =-0.50;
        n[6][0] =-1.00; n[6][1] = 0.00; n[6][2] = 0.00; n[6][3] = 0.00;
        v[7][0] =-0.50; v[7][1] = 0.50; v[7][2] = 0.50; v[7][3] = 0.50;
        n[7][0] =-1.00; n[7][1] = 0.00; n[7][2] = 0.00; n[7][3] = 0.00;
        decompose_hexahedron(v, n);

        v[0][0] = 0.50; v[0][1] = 0.50; v[0][2] = 0.50; v[0][3] = 0.50;
        n[0][0] = 1.00; n[0][1] = 0.00; n[0][2] = 0.00; n[0][3] = 0.00;
        v[1][0] = 0.50; v[1][1] = 0.50; v[1][2] = 0.50; v[1][3] =-0.50;
        n[1][0] = 1.00; n[1][1] = 0.00; n[1][2] = 0.00; n[1][3] = 0.00;
        v[2][0] = 0.50; v[2][1] = 0.50; v[2][2] =-0.50; v[2][3] = 0.50;
        n[2][0] = 1.00; n[2][1] = 0.00; n[2][2] = 0.00; n[2][3] = 0.00;
        v[3][0] = 0.50; v[3][1] = 0.50; v[3][2] =-0.50; v[3][3] =-0.50;
        n[3][0] = 1.00; n[3][1] = 0.00; n[3][2] = 0.00; n[3][3] = 0.00;
        v[4][0] = 0.50; v[4][1] =-0.50; v[4][2] = 0.50; v[4][3] = 0.50;
        n[4][0] = 1.00; n[4][1] = 0.00; n[4][2] = 0.00; n[4][3] = 0.00;
        v[5][0] = 0.50; v[5][1] =-0.50; v[5][2] = 0.50; v[5][3] =-0.50;
        n[5][0] = 1.00; n[5][1] = 0.00; n[5][2] = 0.00; n[5][3] = 0.00;
        v[6][0] = 0.50; v[6][1] =-0.50; v[6][2] =-0.50; v[6][3] = 0.50;
        n[6][0] = 1.00; n[6][1] = 0.00; n[6][2] = 0.00; n[6][3] = 0.00;
        v[7][0] = 0.50; v[7][1] =-0.50; v[7][2] =-0.50; v[7][3] =-0.50;
        n[7][0] = 1.00; n[7][1] = 0.00; n[7][2] = 0.00; n[7][3] = 0.00;
        decompose_hexahedron(v, n);
        gl4DEnd();

        ambient[0] = 0.10; ambient[1] = 0.75; ambient[2] = 0.10; ambient[3] = 1.0;
        diffuse[0] = 0.25; diffuse[1] = 1.00; diffuse[2] = 0.25; diffuse[3] = 1.0;
        specular[0] = 1.00; specular[1] = 1.00; specular[2] = 1.00; specular[3] = 1.0;
        gl4DMaterialfv(ambient, diffuse, specular);
        gl4DBegin(GL4D_TETRAHEDRA);
        v[0][0] = 0.50; v[0][1] =-0.50; v[0][2] = 0.50; v[0][3] = 0.50;
        n[0][0] = 0.00; n[0][1] =-1.00; n[0][2] = 0.00; n[0][3] = 0.00;
        v[1][0] = 0.50; v[1][1] =-0.50; v[1][2] = 0.50; v[1][3] =-0.50;
        n[1][0] = 0.00; n[1][1] =-1.00; n[1][2] = 0.00; n[1][3] = 0.00;
        v[2][0] = 0.50; v[2][1] =-0.50; v[2][2] =-0.50; v[2][3] = 0.50;
        n[2][0] = 0.00; n[2][1] =-1.00; n[2][2] = 0.00; n[2][3] = 0.00;
        v[3][0] = 0.50; v[3][1] =-0.50; v[3][2] =-0.50; v[3][3] =-0.50;
        n[3][0] = 0.00; n[3][1] =-1.00; n[3][2] = 0.00; n[3][3] = 0.00;
        v[4][0] =-0.50; v[4][1] =-0.50; v[4][2] = 0.50; v[4][3] = 0.50;
        n[4][0] = 0.00; n[4][1] =-1.00; n[4][2] = 0.00; n[4][3] = 0.00;
        v[5][0] =-0.50; v[5][1] =-0.50; v[5][2] = 0.50; v[5][3] =-0.50;
        n[5][0] = 0.00; n[5][1] =-1.00; n[5][2] = 0.00; n[5][3] = 0.00;
        v[6][0] =-0.50; v[6][1] =-0.50; v[6][2] =-0.50; v[6][3] = 0.50;
        n[6][0] = 0.00; n[6][1] =-1.00; n[6][2] = 0.00; n[6][3] = 0.00;
        v[7][0] =-0.50; v[7][1] =-0.50; v[7][2] =-0.50; v[7][3] =-0.50;
        n[7][0] = 0.00; n[7][1] =-1.00; n[7][2] = 0.00; n[7][3] = 0.00;
        decompose_hexahedron(v, n);

        v[0][0] =-0.50; v[0][1] = 0.50; v[0][2] =-0.50; v[0][3] =-0.50;
        n[0][0] = 0.00; n[0][1] = 1.00; n[0][2] = 0.00; n[0][3] = 0.00;
        v[1][0] =-0.50; v[1][1] = 0.50; v[1][2] =-0.50; v[1][3] = 0.50;
        n[1][0] = 0.00; n[1][1] = 1.00; n[1][2] = 0.00; n[1][3] = 0.00;
        v[2][0] =-0.50; v[2][1] = 0.50; v[2][2] = 0.50; v[2][3] =-0.50;
        n[2][0] = 0.00; n[2][1] = 1.00; n[2][2] = 0.00; n[2][3] = 0.00;
        v[3][0] =-0.50; v[3][1] = 0.50; v[3][2] = 0.50; v[3][3] = 0.50;
        n[3][0] = 0.00; n[3][1] = 1.00; n[3][2] = 0.00; n[3][3] = 0.00;
        v[4][0] = 0.50; v[4][1] = 0.50; v[4][2] =-0.50; v[4][3] =-0.50;
        n[4][0] = 0.00; n[4][1] = 1.00; n[4][2] = 0.00; n[4][3] = 0.00;
        v[5][0] = 0.50; v[5][1] = 0.50; v[5][2] =-0.50; v[5][3] = 0.50;
        n[5][0] = 0.00; n[5][1] = 1.00; n[5][2] = 0.00; n[5][3] = 0.00;
        v[6][0] = 0.50; v[6][1] = 0.50; v[6][2] = 0.50; v[6][3] =-0.50;
        n[6][0] = 0.00; n[6][1] = 1.00; n[6][2] = 0.00; n[6][3] = 0.00;
        v[7][0] = 0.50; v[7][1] = 0.50; v[7][2] = 0.50; v[7][3] = 0.50;
        n[7][0] = 0.00; n[7][1] = 1.00; n[7][2] = 0.00; n[7][3] = 0.00;
        decompose_hexahedron(v, n);
        gl4DEnd();

        ambient[0] = 0.10; ambient[1] = 0.10; ambient[2] = 0.75; ambient[3] = 1.0;
        diffuse[0] = 0.25; diffuse[1] = 0.25; diffuse[2] = 1.00; diffuse[3] = 1.0;
        specular[0] = 1.00; specular[1] = 1.00; specular[2] = 1.00; specular[3] = 1.0;
        gl4DMaterialfv(ambient, diffuse, specular);
        gl4DBegin(GL4D_TETRAHEDRA);
        v[0][0] =-0.50; v[0][1] =-0.50; v[0][2] =-0.50; v[0][3] =-0.50;
        n[0][0] = 0.00; n[0][1] = 0.00; n[0][2] =-1.00; n[0][3] = 0.00;
        v[1][0] =-0.50; v[1][1] =-0.50; v[1][2] =-0.50; v[1][3] = 0.50;
        n[1][0] = 0.00; n[1][1] = 0.00; n[1][2] =-1.00; n[1][3] = 0.00;
        v[2][0] =-0.50; v[2][1] = 0.50; v[2][2] =-0.50; v[2][3] =-0.50;
        n[2][0] = 0.00; n[2][1] = 0.00; n[2][2] =-1.00; n[2][3] = 0.00;
        v[3][0] =-0.50; v[3][1] = 0.50; v[3][2] =-0.50; v[3][3] = 0.50;
        n[3][0] = 0.00; n[3][1] = 0.00; n[3][2] =-1.00; n[3][3] = 0.00;
        v[4][0] = 0.50; v[4][1] =-0.50; v[4][2] =-0.50; v[4][3] =-0.50;
        n[4][0] = 0.00; n[4][1] = 0.00; n[4][2] =-1.00; n[4][3] = 0.00;
        v[5][0] = 0.50; v[5][1] =-0.50; v[5][2] =-0.50; v[5][3] = 0.50;
        n[5][0] = 0.00; n[5][1] = 0.00; n[5][2] =-1.00; n[5][3] = 0.00;
        v[6][0] = 0.50; v[6][1] = 0.50; v[6][2] =-0.50; v[6][3] =-0.50;
        n[6][0] = 0.00; n[6][1] = 0.00; n[6][2] =-1.00; n[6][3] = 0.00;
        v[7][0] = 0.50; v[7][1] = 0.50; v[7][2] =-0.50; v[7][3] = 0.50;
        n[7][0] = 0.00; n[7][1] = 0.00; n[7][2] =-1.00; n[7][3] = 0.00;
        decompose_hexahedron(v, n);

        v[0][0] = 0.50; v[0][1] = 0.50; v[0][2] = 0.50; v[0][3] = 0.50;
        n[0][0] = 0.00; n[0][1] = 0.00; n[0][2] = 1.00; n[0][3] = 0.00;
        v[1][0] = 0.50; v[1][1] = 0.50; v[1][2] = 0.50; v[1][3] =-0.50;
        n[1][0] = 0.00; n[1][1] = 0.00; n[1][2] = 1.00; n[1][3] = 0.00;
        v[2][0] = 0.50; v[2][1] =-0.50; v[2][2] = 0.50; v[2][3] = 0.50;
        n[2][0] = 0.00; n[2][1] = 0.00; n[2][2] = 1.00; n[2][3] = 0.00;
        v[3][0] = 0.50; v[3][1] =-0.50; v[3][2] = 0.50; v[3][3] =-0.50;
        n[3][0] = 0.00; n[3][1] = 0.00; n[3][2] = 1.00; n[3][3] = 0.00;
        v[4][0] =-0.50; v[4][1] = 0.50; v[4][2] = 0.50; v[4][3] = 0.50;
        n[4][0] = 0.00; n[4][1] = 0.00; n[4][2] = 1.00; n[4][3] = 0.00;
        v[5][0] =-0.50; v[5][1] = 0.50; v[5][2] = 0.50; v[5][3] =-0.50;
        n[5][0] = 0.00; n[5][1] = 0.00; n[5][2] = 1.00; n[5][3] = 0.00;
        v[6][0] =-0.50; v[6][1] =-0.50; v[6][2] = 0.50; v[6][3] = 0.50;
        n[6][0] = 0.00; n[6][1] = 0.00; n[6][2] = 1.00; n[6][3] = 0.00;
        v[7][0] =-0.50; v[7][1] =-0.50; v[7][2] = 0.50; v[7][3] =-0.50;
        n[7][0] = 0.00; n[7][1] = 0.00; n[7][2] = 1.00; n[7][3] = 0.00;
        decompose_hexahedron(v, n);
        gl4DEnd();

        ambient[0] = 0.75; ambient[1] = 0.75; ambient[2] = 0.10; ambient[3] = 1.0;
        diffuse[0] = 1.00; diffuse[1] = 1.00; diffuse[2] = 0.25; diffuse[3] = 1.0;
        specular[0] = 1.00; specular[1] = 1.00; specular[2] = 1.00; specular[3] = 1.0;
        gl4DMaterialfv(ambient, diffuse, specular);
        gl4DBegin(GL4D_TETRAHEDRA);
        v[0][0] = 0.50; v[0][1] = 0.50; v[0][2] = 0.50; v[0][3] =-0.50;
        n[0][0] = 0.00; n[0][1] = 0.00; n[0][2] = 0.00; n[0][3] =-1.00;
        v[1][0] = 0.50; v[1][1] = 0.50; v[1][2] =-0.50; v[1][3] =-0.50;
        n[1][0] = 0.00; n[1][1] = 0.00; n[1][2] = 0.00; n[1][3] =-1.00;
        v[2][0] = 0.50; v[2][1] =-0.50; v[2][2] = 0.50; v[2][3] =-0.50;
        n[2][0] = 0.00; n[2][1] = 0.00; n[2][2] = 0.00; n[2][3] =-1.00;
        v[3][0] = 0.50; v[3][1] =-0.50; v[3][2] =-0.50; v[3][3] =-0.50;
        n[3][0] = 0.00; n[3][1] = 0.00; n[3][2] = 0.00; n[3][3] =-1.00;
        v[4][0] =-0.50; v[4][1] = 0.50; v[4][2] = 0.50; v[4][3] =-0.50;
        n[4][0] = 0.00; n[4][1] = 0.00; n[4][2] = 0.00; n[4][3] =-1.00;
        v[5][0] =-0.50; v[5][1] = 0.50; v[5][2] =-0.50; v[5][3] =-0.50;
        n[5][0] = 0.00; n[5][1] = 0.00; n[5][2] = 0.00; n[5][3] =-1.00;
        v[6][0] =-0.50; v[6][1] =-0.50; v[6][2] = 0.50; v[6][3] =-0.50;
        n[6][0] = 0.00; n[6][1] = 0.00; n[6][2] = 0.00; n[6][3] =-1.00;
        v[7][0] =-0.50; v[7][1] =-0.50; v[7][2] =-0.50; v[7][3] =-0.50;
        n[7][0] = 0.00; n[7][1] = 0.00; n[7][2] = 0.00; n[7][3] =-1.00;
        decompose_hexahedron(v, n);

        v[0][0] =-0.50; v[0][1] =-0.50; v[0][2] =-0.50; v[0][3] = 0.50;
        n[0][0] = 0.00; n[0][1] = 0.00; n[0][2] = 0.00; n[0][3] = 1.00;
        v[1][0] =-0.50; v[1][1] =-0.50; v[1][2] = 0.50; v[1][3] = 0.50;
        n[1][0] = 0.00; n[1][1] = 0.00; n[1][2] = 0.00; n[1][3] = 1.00;
        v[2][0] =-0.50; v[2][1] = 0.50; v[2][2] =-0.50; v[2][3] = 0.50;
        n[2][0] = 0.00; n[2][1] = 0.00; n[2][2] = 0.00; n[2][3] = 1.00;
        v[3][0] =-0.50; v[3][1] = 0.50; v[3][2] = 0.50; v[3][3] = 0.50;
        n[3][0] = 0.00; n[3][1] = 0.00; n[3][2] = 0.00; n[3][3] = 1.00;
        v[4][0] = 0.50; v[4][1] =-0.50; v[4][2] =-0.50; v[4][3] = 0.50;
        n[4][0] = 0.00; n[4][1] = 0.00; n[4][2] = 0.00; n[4][3] = 1.00;
        v[5][0] = 0.50; v[5][1] =-0.50; v[5][2] = 0.50; v[5][3] = 0.50;
        n[5][0] = 0.00; n[5][1] = 0.00; n[5][2] = 0.00; n[5][3] = 1.00;
        v[6][0] = 0.50; v[6][1] = 0.50; v[6][2] =-0.50; v[6][3] = 0.50;
        n[6][0] = 0.00; n[6][1] = 0.00; n[6][2] = 0.00; n[6][3] = 1.00;
        v[7][0] = 0.50; v[7][1] = 0.50; v[7][2] = 0.50; v[7][3] = 0.50;
        n[7][0] = 0.00; n[7][1] = 0.00; n[7][2] = 0.00; n[7][3] = 1.00;
        decompose_hexahedron(v, n);
        gl4DEnd();
    } else if (g_display_mode == 1) {
        gl4DBegin(GL4D_TETRAHEDRA);
        {
            GLuint i, j, k, l, m;

            for (i = 0; i < g_mesh_w - 1; i++) {
                for (j = 0; j < g_mesh_h - 1; j++) {
                    for (k = 0; k < g_mesh_d - 1; k++) {
                        for (l = 0; l < 8; l++) {
                            GLuint di = (l>>2)&1;
                            GLuint dj = (l>>1)&1;
                            GLuint dk = (l>>0)&1;

                            for (m = 0; m < 4; m++) {
                                v[l][m] = g_mesh_v[i+di][j+dj][k+dk][m];
                                n[l][m] = g_mesh_n[i+di][j+dj][k+dk][m];
                            }
                        }

                        decompose_hexahedron(v, n);
                    }
                }
            }
        }
        gl4DEnd();
    } else if (g_display_mode == 2) {
        gl4DRenderSurface(g_gl4d_surface);
    }

    gl4DRenderEnd();

    glColor4f(1.0f, 1.0f, 1.0f, 1.0f);

    display_fps();
    {
        char buffer[256];

        snprintf(buffer, sizeof(buffer), "%s (%s)",
            g_display_mode_string[g_display_mode],
            g_high_quality ? "High quality" : "Normal quality"
        );

        render_bitmap_string(
            -0.95f, 0.85f, 0.0f, GLUT_BITMAP_HELVETICA_12, buffer);
    }

    glFlush();
    glutSwapBuffers();
}

void rb_3d(int x, int y) {
    gl4d_rb_3d(&g_rb_state_3d, (double) x, (double) -y);
}

void rb_4d_xyw(int x, int y) {
    gl4d_rb_4d(&g_rb_state_4d, (double) x, (double) -y, 0.0);
}

void rb_4d_xzw(int x, int y) {
    gl4d_rb_4d(&g_rb_state_4d, (double) x, 0.0, (double) -(-y));
}

void mouse(int button, int state, int x, int y) {
    int modifiers = glutGetModifiers();

    gl4d_rb_state_reset(&g_rb_state_3d);
    gl4d_rb_state_reset(&g_rb_state_4d);

    if (modifiers == GLUT_ACTIVE_CTRL) {
        g_active_rb_func = rb_4d_xyw;
    } else if (modifiers == (GLUT_ACTIVE_CTRL | GLUT_ACTIVE_SHIFT)) {
        g_active_rb_func = rb_4d_xzw;
    } else {
        g_active_rb_func = rb_3d;
    }

    g_active_rb_func(x, y);
}

void motion(int x, int y) {
    g_active_rb_func(x, y);
    glutPostRedisplay();
}

void keyboard(unsigned char key, int x, int y) {
    switch (key) {
        case 'q':
            g_high_quality = !g_high_quality;
            gl4DParamui(GL4D_PARAM_SLICE, g_high_quality ? 256 : 128);
            break;

        case ' ':
            g_display_mode = (g_display_mode + 1) % g_max_display_mode;
            break;
    }
}

#ifdef IDLE_REDRAW
void idle(void) {
    glutSetWindow(g_main_window);
    glutPostRedisplay();
}
#endif

double light_4d_get_radius() {
    double radius;

    if (g_light_4d_pos[2] > 1.0 - EPSILON) {
        radius = 0.0;
    } else {
        radius = sqrt(1.0 - g_light_4d_pos[2] * g_light_4d_pos[2]);
    }

    return radius;
}

void light_4d_reshape(int w, int h) {
    glViewport(0, 0, w, h);

    g_light_4d_screen[0] = w;
    g_light_4d_screen[1] = h;
}

void light_4d_display(void) {
    glClear(GL_COLOR_BUFFER_BIT);

    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glOrtho(-1.0, 1.0, -1.0, 1.0, -1.0, 1.0);

    /* small sphere */
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glTranslatef(0.7, 0.7, 0.0);
    glScalef(0.25, 0.25, 0.25);

    glPushMatrix();
    glRotatef(90.0, 1.0, 0.0, 0.0);

    glColor4f(0.25, 0.25, 0.25, 1.0);
    glutWireSphere(1.0, 20, 20);

    glPopMatrix();

    glColor4f(1.0, 1.0, 1.0, 1.0);
    glBegin(GL_LINES);
        glVertex3d(-1.1, g_light_4d_pos[2], 0.0);
        glVertex3d( 1.1, g_light_4d_pos[2], 0.0);
    glEnd();

    glPopMatrix();

    /* large sphere */

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glTranslatef(-0.125, -0.125, 0.0);
    glScalef(0.85, 0.85, 0.85);

    glColor4f(1.0, 1.0, 1.0, 1.0);
    glBegin(GL_TRIANGLE_FAN);
    {
        const unsigned int segments = 180;

        double radius = light_4d_get_radius();
        unsigned int i;

        glVertex2d(0.0, 0.0);

        for (i = 0; i <= segments; i++) {
            double theta = (double) i / (double) segments * M_PI * 2.0;

            glVertex2d(sin(theta) * radius, cos(theta) * radius);
        }
    }
    glEnd();

    glColor4f(0.25, 0.25, 0.25, 1.0);
    glutWireSphere(1.0, 20, 20);

    glPointSize(5.0f);
    glColor4f(1.0f, 0.0f, 0.0f, 1.0f);

    glBegin(GL_POINTS);
        glVertex3dv(g_light_4d_pos);
    glEnd();

    glPopMatrix();

    /* OSD */
    {
        char buffer[255];
        snprintf(buffer, sizeof(buffer),
            "Diretional Light #0 = (%.2f, %.2f, %.2f, %.2f)",
            g_light_4d_pos[0],
            g_light_4d_pos[1],
            g_light_4d_pos[2],
            g_light_4d_pos[3]
        );

        glColor4d(1.0, 1.0, 1.0, 1.0);
        render_bitmap_string(
            -0.95f, 0.9f, 0.0f, GLUT_BITMAP_HELVETICA_12, buffer);
    }

    glMatrixMode(GL_PROJECTION);
    glPopMatrix();

    glutSwapBuffers();
}

void light_4d_mouse(int button, int state, int x, int y) {
    g_light_control_xy = button == GLUT_LEFT_BUTTON;
}

void light_4d_motion(int x, int y) {
    GL4Ddouble pos[2];
    GL4Ddouble length;
    GL4Ddouble radius;

    pos[0] =  ((double) x / (double) g_light_4d_screen[0] * 2.0 - 1.0);
    pos[1] = -((double) y / (double) g_light_4d_screen[1] * 2.0 - 1.0);

    if (g_light_control_xy) {
        g_light_4d_pos[0] = pos[0];
        g_light_4d_pos[1] = pos[1];
    } else {
        if (pos[1] < -1.0 || pos[1] > 1.0) {
            pos[1] = copysign(1.0, pos[1]);
        }

        g_light_4d_pos[2] = pos[1];
    }

    length = sqrt(
        g_light_4d_pos[0] * g_light_4d_pos[0] +
        g_light_4d_pos[1] * g_light_4d_pos[1]
    );

    radius = light_4d_get_radius();

    if (length >= radius) {
        if (length < EPSILON) {
            g_light_4d_pos[0] = 0.0;
            g_light_4d_pos[1] = 0.0;
        } else {
            g_light_4d_pos[0] = g_light_4d_pos[0] / length * radius;
            g_light_4d_pos[1] = g_light_4d_pos[1] / length * radius;
        }
        g_light_4d_pos[3] = 0.0;
    } else {
        g_light_4d_pos[3] = sqrt(
            1.0 -
            g_light_4d_pos[0] * g_light_4d_pos[0] -
            g_light_4d_pos[1] * g_light_4d_pos[1] -
            g_light_4d_pos[2] * g_light_4d_pos[2]
        );
    }

    g_light_4d_pos[3] = -g_light_4d_pos[3];

    glutPostRedisplay();
}

int main(int argc, char **argv) {
    fprintf(stderr, "========================================================================\n");
    fprintf(stderr, "  GL4D API Demo\n");
    fprintf(stderr, "========================================================================\n");
    fprintf(stderr, "Keyboard control for main window: \n");
    fprintf(stderr, "                   spacebar : Cycle among the following display modes: \n");
    fprintf(stderr, "                              - Immediate mode hypercube\n");
    fprintf(stderr, "                              - Immediate mode 4D torus\n");
    fprintf(stderr, "                              - Retained mode 4D torus\n");
    fprintf(stderr, "  \n");
    fprintf(stderr, "                          q : Toggle between high/normal quality\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Mouse control for main window: \n");
    fprintf(stderr, "                 mouse drag : 3D model view matrix\n");
    fprintf(stderr, "          ctrl + mouse drag : 4D model view matrix (xyw)\n");
    fprintf(stderr, "  ctrl + shift + mouse drag : 4D model view matrix (xzw)\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Mouse control for 4D light control window: \n");
    fprintf(stderr, "            left mouse drag : Modify xy components\n");
    fprintf(stderr, "           right mouse drag : Modify z component\n");
    fprintf(stderr, "========================================================================\n");
    fprintf(stderr, "\n");

    glutInit(&argc, argv);
    glutInitDisplayMode(GLUT_DEPTH | GLUT_DOUBLE | GLUT_RGBA);

    /* light position */
    glutInitWindowPosition(100 + 500 + 10, 100);
    glutInitWindowSize(500, 500);

    g_light_4d_window = glutCreateWindow("4D Light Control");
    glutReshapeFunc(light_4d_reshape);
    glutDisplayFunc(light_4d_display);
    glutMouseFunc(light_4d_mouse);
    glutMotionFunc(light_4d_motion);

    /* main window */
    glutInitWindowPosition(100, 100);
    glutInitWindowSize(500, 500);

    g_main_window = glutCreateWindow("GL4D Demo");
    glutReshapeFunc(reshape);
    glutDisplayFunc(display);
    glutMouseFunc(mouse);
    glutMotionFunc(motion);
    glutKeyboardFunc(keyboard);

#ifdef IDLE_REDRAW
    glutIdleFunc(idle);
#endif

    if (gl4DInit() != GL4D_NO_ERROR) {
        fprintf(stderr, "GL4D initialization failed. \n");

        return 0;
    }

    gl4d_rb_state_init(&g_rb_state_3d);
    gl4d_rb_state_init(&g_rb_state_4d);

    load_mesh("4dtorus19.mnf");

    glutMainLoop();

    return 0;
}

