function [G,err,att]=lowpass_filter_design(n,p,q,nus,K)
%%%
%%% TITLE: lowpass_filter_design
%%%
%%% PROVIDES: an orthonormal low-pass filter G to be used in 
%%% a rational p/q branch
%%%
%%% USAGE: [G,err,att] = lowpass_filter_design(n,p,q,nus,K)
%%% where
%%%    n   : degree of the filter. It is advised to take n=n0*p-1
%%%    p,q : down- and up-sampling factors p>q
%%%    nus : beginning of the stop-band (normalized frequency) >1/(2*p)
%%%    K   : regularity order
%%%    err : optional orthonormality accuracy for each iteration step
%%%    att : optional L2 measure of the attenuation for each iteration step
%%%
%%% NOTES: the accuracy of the orthonormality can be modified by changing  
%%% the value of the variable "orthonormality_accuracy". In practice, 
%%% if n is not of the form n0*p-1, the coefficients of G of highest power
%%% between n and n0*p-1 will be found to be below the orthonormality accuracy, 
%%% i.e., they can be set to 0.
%%% If the regularity order K=0 or 1, then the algorithm converges for all 
%%% values of (p,q), up to the accuracy of the computer. If K>=2, the algorithm
%%% is ensured to converge only for integer scaling factors, i.e., q=1, p arbitrary.
%%% A way to compute Daubechies-like filters is to set n=0 and to choose the 
%%% regularity factor K: the algorithm then automatically computes the 
%%% minimum degree of the filter that complies with the constraints.
%%%
%%% REFERENCE: T. Blu, "A new design algorithm for two-band orthonormal rational 
%%% filter banks and orthonormal rational wavelets," IEEE Trans. Signal Processing, 
%%% vol. 46, no. 6, pp. 1494-1504, June 1998.
%%%
%%% AUTHOR: Thierry BLU, Swiss Federal Institute of Technology, Lausanne (EPFL)
%%%
%%% VERSION: 1993-1998
%%%

orthonormality_accuracy=1e-10;
%%%%%% are there too many constraints (orthonormality, regularity) 
%%%%%% w/respect to the degree of the filter
q2=q*q;
e=rem(q-1+K*(p-1),p-q);
n0=e+(p-q)*floor((p-1-e)/(p-q));
n1=floor((q-1+K*(p-1)-n0)/(p-q));
if n1<0
  n1=0;
  n0=q-1+K*(p-1);
end
nn=n0+n1*p;
if n<nn
  disp(sprintf('! n cannot be smaller than %0.0f (new value of n : %0.0f)',nn,nn))
  n=nn;
end

%%%%%% scalar product: fixed part
alpha0=1;                  %%%%%% weight of A0 (attenuation)
alpha1=0;                  %%%%%% weight of A1 (error Gn-Gn1)
alpha2=1;                  %%%%%% weight of A2 (reconstruction)
alpha3=0;                  %%%%%% weight of A3 (regularity)
A0=alpha0*scalar_product(n,nus,0.5); 
A1=alpha1*eye(size(A0));         %%%%%% error between successive iterations

%%%%%% regularity
nn=0:n;
T=[];
if K~=0
  T(K*(p-1),1+n)=0;
end
for k=0:(K-1)
  for s=0:(p-2)
    T(1+s+k*(p-1),1+nn)=-(nn.^k);
    T(1+s+k*(p-1),1+(s:p:n))=T(1+s+k*(p-1),1+(s:p:n))*(1-p);
  end
end
T=round(T);
for k=1:(K*(p-1))
  T(k,:)=T(k,:)/max(T(k,:));
end
TT=[];
if q>=2
  if K~=0
    TT(K*(q-1),1+n)=0;
  end
  for k=0:(K-1)
    for s=0:(q-2)
      TT(1+s+k*(q-1),1+nn)=-(nn.^k);
      TT(1+s+k*(q-1),1+(s:q:n))=TT(1+s+k*(q-1),1+(s:q:n))*(1-q);
    end
  end
  TT=round(TT);
  for k=1:(K*(q-1))
    T(k,:)=T(k,:)/max(T(k,:));
  end
end

T=[T;TT];

if K==0
  A3=zeros(n+1,n+1);
else
	A3=alpha3*T.'*T;
end

%%%%%% computation of u[n0,n'0] and U[n0,n'0] allowing to get 
%%%%%% in particular the number of independent equations
u=[];U=[];V=[];
for n0=0:q-1
  for nn0=0:q-1
    u(n0+nn0*q+1)=1+floor((floor((n+nn0*p)/q)-ceil(n0*p/q))/p);
  end
  V(n0+1)=n0*(n0+1)/2;
end
U=[];U(q*q)=0;
for k=1:q*q
  U(k+1)=U(k)+u(k)-1;
end
kj=sum(u)-q*q;                  
ks=sum(u)-q*(q-1)/2;            

kss=ks+K*(p-1);

r=[1:kss];
ss=[1:ks];

%%%%%% computation of the constant term C
C=[];
C(kss,1)=0;
for n0=0:q-1
  C(1+kj+V(n0+1)+n0,1)=1;
end

Cs=C(ss,1);
Cr=C(r,1);


%%%%%% initialization of Gn with a random filter
gamma=puiss(conv(ones(1,p),ones(1,q)),K);
Gn=conv(gamma,rand(1,n+2-length(gamma))).'; 

errmin=1;
Gmin=Gn;

err=[];
encore=1;                   
while encore


  Giter=rev(Gn.');

  %%%%%% computation of the reconstruction error
  for s=0:floor(q/2)
    Hn=abs(conv(rev(Giter),Ha(Giter,exp(2*i*pi*s/q))));
    debut=1+rem(n,p);
    if s==0
      erreur=max([Hn(debut:p:n) abs(q-Hn(n+1)) Hn((n+1+p):p:(2*n+1))]);
    else
      erreur=max([erreur Hn(debut:p:2*n+1)]);
    end
  end
  fprintf('orthonormality accuracy = %e\n',erreur)
  err=[err erreur];

  %%%%%% computation of the constraint matrix S
  S=[];  
  S(ks,n+1)=0;
  for n0=0:q-1
    for nn0=0:q-1
      for s=1:(u(n0+q*nn0+1)-1)
        jmin=rem(q-rem(n0*p,q),q)+p*(n0-nn0)+p*q*s;
        S(U(n0+q*nn0+1)+s,1+(jmin:q:n))=Gn(1-p*(n0-nn0+s*q)+(jmin:q:n),1).';
      end
    end
  end

  for n0=0:q-1
    for nn0=0:n0
      jmin=rem(q-rem(n0*p,q),q)+p*(n0-nn0);
      S(1+kj+V(n0+1)+nn0,1+(jmin:q:n))=Gn(1-p*(n0-nn0)+(jmin:q:n),1).';

    end
  end
  
  S=[S;T];

  %%%%%% computation of the scalar product
  Sr=S(r,:);
  Ss=S(ss,:);

  A2=alpha2*Ss.'*Ss;

  M=A0+A1+A2+A3;
  MM=Sr*(M\Sr.');

  %%%%%% computation of Gn
  GGn=M\(alpha1*Gn+alpha2*Ss.'*Cs);
  lambda=MM\(Cr-Sr*GGn);
  Gna=GGn+M\(Sr.'*lambda);
  
  %%%%%% keep only the minimal Gn 
  if erreur<errmin
	  errmin=erreur;
	  Gmin=Gn;
  end

  %%%%%% induction
  Gn=(Gn+Gna)/2;        

  %%%%%% test for continuing
  encore=(erreur>orthonormality_accuracy&length(err)<500);

end

att=Gmin'*A0*Gmin/alpha0;
G=rev(Gmin.');
fprintf('\nfinal accuracy       = %e\n',errmin)
fprintf(  'number of iterations : %g\n',length(err))

%%%%
%%%% END OF THE MAIN PROGRAM
%%%%

%%%%
%%%% OTHER FUNCTIONS
%%%%

function M=scalar_product(n,nu1,nu2)
%
% M=scalar-product(n,nu1,nu2)
% n   = degree of the filter
% nu1,
% nu2 = beginning and end of the attenuation frequencies
%

k=1:n;
W=[];
W(n+1)=0;

W(1)=(nu2-nu1);
W(1,2:n+1)=(sin(2*k*pi*nu2)-sin(2*k*pi*nu1))./k/2/pi;

M=[];
M(n+1,n+1)=0;
M(1,1:n+1)=W(1:n+1);
M(n+1,1:n+1)=W(n+1:-1:1);
	
for k=1:n-1
	M(k+1,k+1:n+1)=W(1:n+1-k);
	M(k+1,1:k)=W(k+1:-1:2);
end

%%%
%%%

function y=puiss(g,n)
y=[1];
while n>0,
	if rem(n,2)==1
		y=conv(g,y);
	end
	g=conv(g,g);
	n=fix(n/2);
end

%%%
%%%

function H=rev(G)
H=G(length(G):-1:1);

%%%
%%%

function G=Ha(H,a)
u=a.^(length(H)-1:-1:0);
G=u.*H;
