function w=FFTwaveletanalysis1D(x,FFTanalysisfilters,J);
% 	Usage: w=FFTwaveletanalysis1D(x,FFTanalysisfilters,J) 
%
%   FFT-based implementation of the wavelet transform (see ref. [2]). It computes the wavelet transform of a 
%   signal "x" using a Fourier method. It uses periodic boundary conditions.  The length of the signal 
%   must be a power of two and the frequency responses of the filters is specified in FFTanalysisfilters.
% 	
% 	Input:
% 	x = input signal, of size 2^N=length of FFTanalysisfilters
% 	J = depth of the decomposition, i.e., J wavelet bands (wav1 to wavJ) 
% 	    + 1 lowpass (lowJ) 
%   FFTanalysisfilters = [lowpassfilter;highpassfilter]
% 	
% 	Output:
% 	w = [wav1 wav2 ...  wavJ lowJ]: vector of pooled wavelet coefficients, size 2^N
% 	 	
% 	See also FFTWAVELETSYNTHESIS1D, FFTFRACTSPLINEFILTERS, WEXTRACT1D.
% 	
% 	Author: Thierry Blu
% 	The Chinese University of Hong Kong.
% 	This software is downloadable at http://www.ee.cuhk.edu.hk/~tblu/demos/
% 	
% 	References:
% 	[1] Unser, M. & Blu, T.,"Fractional Splines and Wavelets", SIAM Review, Vol. 42 (1), pp. 43-67, March 2000. 
% 	[2] Blu, T. & Unser, M.,"The Fractional Spline Wavelet Transform: Definition and Implementation", Proc. IEEE International 
%       Conference on Acoustics, Speech, and Signal Processing (ICASSP'00), Istanbul, Turkey, Vol. {I}, pp. 512-515, June 5--9, 2000.  
%   [3] Blu, T. & Unser, M.,"A Complete Family of Scaling Functions: The $($-Fractional Splines", Proc. IEEE International 
%       Conference on Acoustics, Speech, and Signal Processing (ICASSP'03), Hong Kong, China, Vol. {VI}, pp. 421-424, April 6--10, 2003. 

M=length(x);
if M~=2^round(log(M)/log(2))
	disp(' ')
	disp('The size of the input signal must be a power of two!')
	disp(' ')
	w=[];
	return
end

if length(FFTanalysisfilters)~=M
	disp(' ')
	disp('The size of the input signal and of the filters must match!')
	disp(' ')
	w=[];
	return
end

% Fourier transform of the signal
X=fft(x,M);

G=FFTanalysisfilters(1,:);
H=FFTanalysisfilters(2,:);

w=[];
for j=1:J
	%
	% Computation of the outputs y and z
	%
	Y=G.*X;
	Z=H.*X;
	Y=1/2*(Y(1:M/2)+Y(M/2+(1:M/2)));
	Z=1/2*(Z(1:M/2)+Z(M/2+(1:M/2)));
	z=ifft(Z,M/2);
	w=[w z];
	
	M=M/2;
	X=Y;
	G=G(1:2:length(G));
	H=H(1:2:length(H));
end
w=real([w ifft(X,M)]);

