function [h_c,h]=convKernel(im_size,type,K)
%[H_C,H]=CONVKERNEL(IM_SIZE,TYPE)
% 
% Generate different convolution kernal in spatial domain depends on
% the input 'type'. User-defined kernals are possible by either directly
% modifing the spatial domain definitions of the PSF or load a *.mat file
% with the input argument 'type' being 4.
% 
% Input: im_size: the size of image that is to be blurred
%        type: convolution kernel types, incluidng
%        1: a 9 x 9 uniform blur
%        2: 15 x 15 blur with h_{i,j}=1/(1+i^2+j^2), i,j=-7,...7
%        3: 5 x 5 seperable spline blur: [1,4,6,4,1]/16
%        4: load user specified .mat file to define the convolution kernel
% 
% Output: h_c: convolution filter that is centered to the upper-left corner
%         h: convoultion kernel that's not centered
% 
% Written by Hanjie Pan, 2/8/2010
% 
switch type
    case 1
        h=ones(9);
    case 2
        [I,J]=meshgrid(-7:7,-7:7);
        h=1./(1+I.^2+J.^2);
    case 3
        h=[1;4;6;4;1]*[1 4 6 4 1];
    case 4
        dir=fileparts(mfilename('fullpath'));
        bk=0;
        while ~bk
        [filename, pathname]=uigetfile('*.mat',...
            'Select Customized Point-Spread Function (PSF) File',...
            [dir(1:end-5),'psf/psf1.mat']);
        if isequal(filename,0)
            bk=0;
        else
            bk=1;
            load(fullfile(pathname, filename));
        end
        end
    case 5
        % uniform blur with various filter size
        if nargin<3
            K=9;
        end
        h=ones(ceil(K));
    case 6
        % Gaussian blur with various stand deviation
        if nargin<3
            K=3;
        end
        hsize=floor(3*K)*2+1;
        h=fspecial('gaussian', hsize, K);
    otherwise
        error('Unrecognized blurring type.');
end
% normalization
h=h./sum(h(:));
% shift the center of 'h' to (1,1), i.e. the upper left corner of the image plane
h_c=zeros(im_size);
h_c(1:size(h,1),1:size(h,2))=h;
h_c=circshift(h_c,[-floor(size(h,1)/2),-floor(size(h,2)/2)]);
return