%ILET_DEMO1.M 
% Deconvolution by minimizing the l1-regularized objective function J(c):
% 
% 	min_c	|c|_1
% 	s.t.	|y-HWc|_2^2 <= noise energy
% 
% with decimated wavelet transform. Here 'H' is the PSF convolution matrix,
% 'W' is the (decimated) wavelet synthesis operator, and 'c' are the 
% wavelet coefficients such that the spatial domain image x = Wc. The 
% constrained problem is converted to the associated Lagrangian
% 
%   min_c   |y-HWc| + lambda |c|_1
% 
% where the regularization weight 'lambda' is updated automatically such 
% that the reconstruction 'c_star' is compatible with the noise level: 
% |y-HWc_star|^2 = noise energy.
% 
% Authors: Hanjie Pan and Thierry Blu, October 2011
% Image and Video Processing Lab, CUHK, Shatin, N.T., Hong Kong
% This software is available at http://www.ee.cuhk.edu.hk/~hjpan
% 
% References:
% 
% [1] H. Pan and T. Blu, "Sparse image restoration using iterated linear 
% expansion of thresholds," in 2011 IEEE International Conference on Image 
% Processing (ICIP2011), (Brussels, Belgium), September 2011.
% 
% [2] H. Pan and T. Blu, "An iterated linear expansion of thresholds for 
% l1-based image restoration." Submitted to IEEE Transactions on Image 
% Processing.
% 
clear;close all;%clc;
dir=fileparts(mfilename('fullpath'));
addpath([dir,'/src'])              	% source files for all three comparing algorithms
addpath(genpath([dir,'/tools']))   	% 'tools' include rice wavelet toolbox
addpath([dir,'/image'])            	% where the test images are stored

%% ------- parameters -------
im_name='cameraman.tif';           	% test image name

blur_type=1;                       	% blur type --- 1: 9 x 9 uniform blur; 
                                  	% 2: 1/(1+i^2+j^2) for i,j=-7:7; 
                                    % 3: [1 4 6 4 1]^T [1 4 6 4 1];
                                    % 4: Customized PSF by loading a *.mat file

BSNR_db=40;                        	% blurred signal to noise ratio (in dB)

%% Distort (blur and add noise) original image
im=double(imread(im_name));         % input image
[N,M]=size(im);                     % image size

% blurring
PSF=convKernel([N,M],blur_type);   	% create convolution kernel
                                    % Here 'PSF' is zero padded and centered at the 
                                    % upper-left corner of the image
y=convolve2(im,PSF);
sigma=norm(y-mean(y(:)),'fro')/sqrt(N*M*10^(0.1*BSNR_db));
randn('state',[362436069;521288629]);% to have reproducible results
% blurred and noisy measurement
noise=sigma.*randn(N,M);
blurred=convolve2(im,PSF)+noise;

psnr_0=PSNR(blurred,im);%

%% -- i-LET deconvolution --
t0=tic;
deblurred=iLET_deconv(blurred,PSF);
t_ilet=toc(t0);

fprintf('Finished!\n');
psnr_ilet=PSNR(deblurred,im);

% ------- plot results -------
h1=figure(1);
set(h1,'name','i-LET Unconstrained Formulation Demo');
set(h1,'Units','normalized');
set(h1,'Position',[0,0.3,1,0.5]);
xmax=2^(nextpow2(max(y(:))))-1;
subplot(131),imagesc(im,[0 xmax]);
axis image;axis off;colormap gray;
title('{\bf Original}','FontSize',14,'interpreter','latex')
subplot(132),imagesc(blurred,[0 xmax]);
axis image;axis off;colormap gray;
title(['{\bf Blurred: PSNR = ',num2str(psnr_0,'%.2f'),'dB}'],...
    'FontSize',14,'interpreter','latex')
subplot(133),imagesc(deblurred,[0 xmax]);
axis image;axis off;colormap gray;
title(['{\bf Deblurred: PSNR = ',num2str(psnr_ilet,'%.2f'),'dB}'],...
    'FontSize',14,'interpreter','latex')
fprintf('Input PSNR  : %.2f[dB]\n',psnr_0);
fprintf('Output PSNR : %.2f[dB]\n',psnr_ilet);
fprintf('Time Elapsed: %5.2f[sec]\n',t_ilet);