#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>


#include "crypto.h"
#include "rsa.h"



void CRYPTO_InitKey(CRYPTO_KEY *key)
{
	key->type = CRYPTO_KEY_UNDEFINED;
	key->num_param = 0;
	key->param = NULL;
}

void CRYPTO_InitKeyPair(CRYPTO_KEY_PAIR *key_pair)
{
	key_pair->type = CRYPTO_KEY_UNDEFINED;
	
	key_pair->public_key.type = CRYPTO_KEY_UNDEFINED;
	key_pair->public_key.num_param = 0;
	key_pair->public_key.param = NULL;

	key_pair->private_key.type = CRYPTO_KEY_UNDEFINED;
	key_pair->private_key.num_param = 0;
	key_pair->private_key.param = NULL;
}

void CRYPTO_DestroyKey(CRYPTO_KEY *key)
{
	int i;
	
	for (i=0; i<key->num_param; i++) {
		mpz_clear(key->param[i]);
	}
	
	if (key->param) {
		free(key->param);
	}
	
	key->type = CRYPTO_KEY_UNDEFINED;
	key->num_param = 0;
	key->param = NULL;
}

void CRYPTO_DestroyKeyPair(CRYPTO_KEY_PAIR *key_pair)
{
	int i;
	
	for (i=0; i<key_pair->public_key.num_param; i++) {
		mpz_clear(key_pair->public_key.param[i]);
	}

	for (i=0; i<key_pair->private_key.num_param; i++) {
		mpz_clear(key_pair->private_key.param[i]);
	}
	
	if (key_pair->public_key.param) {
		free(key_pair->public_key.param);
	}
	
	if (key_pair->private_key.param) {
		free(key_pair->private_key.param);
	}		
	
	key_pair->public_key.type = CRYPTO_KEY_UNDEFINED;
	key_pair->public_key.num_param = 0;
	key_pair->public_key.param = NULL;

	key_pair->private_key.type = CRYPTO_KEY_UNDEFINED;
	key_pair->private_key.num_param = 0;
	key_pair->private_key.param = NULL;
}

BOOL CRYPTO_NewKeyPair(CRYPTO_KEY_PAIR *key_pair, unsigned int bit_length, unsigned int pub_exp, unsigned int key_type)
{
	switch(key_type) {
	case CRYPTO_KEY_RSA:
		return RSA_NewKeyPair(key_pair, bit_length, pub_exp);
	default:
		return FALSE;
	}
}

BOOL CRYPTO_LoadKeyPair(CRYPTO_KEY_PAIR *key_pair, char key_filename[], char passwd[], unsigned int key_type)
{
	switch(key_type) {
	case CRYPTO_KEY_RSA:
		return RSA_LoadKeyPair(key_pair, key_filename, passwd);
	default:
		return FALSE;
	}
}

BOOL CRYPTO_SaveKeyPair(CRYPTO_KEY_PAIR key_pair, char key_filename[], char passwd[])
{
	switch(key_pair.type) {
	case CRYPTO_KEY_RSA:
		return RSA_SaveKeyPair(key_pair, key_filename, passwd);
	default:
		return FALSE;
	}
}

CRYPTO_KEY* CRYPTO_CopyKey(CRYPTO_KEY *dest, CRYPTO_KEY src)
{
	int i;
	
	// release previous allocation
	if (dest->param) {
		for (i=0; i<dest->num_param; i++) {
			mpz_clear(dest->param[i]);
		}
	}
	
	// reallocate to the new size
	dest->param = (mpz_t*)realloc(dest->param, sizeof(mpz_t) * src.num_param);
	if (dest->param == NULL) {
		return NULL;
	}
	
	// initialize the newly allocated mpz_t struct
	for (i=0; i<src.num_param; i++) {
		mpz_init(dest->param[i]);
	}

	// copy the key
	dest->type = src.type;
	dest->num_param = src.num_param;
	for (i=0; i<src.num_param; i++) {
		mpz_set(dest->param[i], src.param[i]);
	}
	
	return dest;
}

CRYPTO_KEY* CRYPTO_CopyPublicKey(CRYPTO_KEY *dest, CRYPTO_KEY_PAIR src)
{
	int i;

	// release previous allocation
	if (dest->param) {
		for (i=0; i<dest->num_param; i++) {
			mpz_clear(dest->param[i]);
		}
	}
	
	// reallocate to the new size
	dest->param = (mpz_t*)realloc(dest->param, sizeof(mpz_t) * src.public_key.num_param);
	if (dest->param == NULL) {
		return NULL;
	}
	
	// initialize the newly allocated mpz_t struct
	for (i=0; i<src.public_key.num_param; i++) {
		mpz_init(dest->param[i]);
	}
	
	// copy the key
	dest->type = src.public_key.type;
	dest->num_param = src.public_key.num_param;
	for (i=0; i<src.public_key.num_param; i++) {
		mpz_set(dest->param[i], src.public_key.param[i]);
	}
	
	return dest;
}

CRYPTO_KEY* CRYPTO_CopyPrivateKey(CRYPTO_KEY *dest, CRYPTO_KEY_PAIR src)
{
	int i;
	
	// release previous allocation
	if (dest->param) {
		for (i=0; i<dest->num_param; i++) {
			mpz_clear(dest->param[i]);
		}
	}
	
	// reallocate to the new size
	dest->param = (mpz_t*)realloc(dest->param, sizeof(mpz_t) * src.private_key.num_param);
	if (dest->param == NULL) {
		return NULL;
	}
	
	// initialize the newly allocated mpz_t struct
	for (i=0; i<src.private_key.num_param; i++) {
		mpz_init(dest->param[i]);
	}

	// copy the key	
	dest->type = src.private_key.type;
	dest->num_param = src.private_key.num_param;
	for (i=0; i<src.private_key.num_param; i++) {
		mpz_set(dest->param[i], src.private_key.param[i]);
	}
	
	return dest;
}

char* CRYPTO_KeyToString(char *result, CRYPTO_KEY key)
{
	int i, offs;
	
	sprintf(result, "%d %d ", key.type, key.num_param);
	offs = strlen(result);
	
	for (i=0; i<key.num_param; i++) {
		gmp_sprintf(result+offs, "%Zd ", key.param[i]);
		offs += strlen(result+offs);
	}
	
	result[offs-1] = '\0';
	
	return result;
}

char* CRYPTO_KeyPairToString(char *result, CRYPTO_KEY_PAIR key_pair)
{
	int i, offs;

	sprintf(result, "%d %d %d ", key_pair.type, key_pair.public_key.type, key_pair.public_key.num_param);
	offs = strlen(result);

	for (i=0; i<key_pair.public_key.num_param; i++) {
		gmp_sprintf(result+offs, "%Zd ", key_pair.public_key.param[i]);
		offs += strlen(result+offs);
	}

	sprintf(result+offs, "%d %d ", key_pair.private_key.type, key_pair.private_key.num_param);
	offs += strlen(result+offs);

	for (i=0; i<key_pair.private_key.num_param; i++) {
		gmp_sprintf(result+offs, "%Zd ", key_pair.private_key.param[i]);
		offs += strlen(result+offs);
	}

	result[offs-1] = '\0';

	return result;
}

int CRYPTO_KeyToStringAlloc(char **result, CRYPTO_KEY key)
{
	int len = CRYPTO_GetKeyStringLength(key);
	
	*result = (char*)realloc(*result, sizeof(char) * len);
	
	CRYPTO_KeyToString(*result, key);
	
	return len;
}

int CRYPTO_KeyPairToStringAlloc(char **result, CRYPTO_KEY_PAIR key_pair)
{
	int len = CRYPTO_GetKeyPairStringLength(key_pair);

	*result = (char*)realloc(*result, sizeof(char) * len);
	
	CRYPTO_KeyPairToString(*result, key_pair);

	return len;
}

int CRYPTO_GetKeyStringLength(CRYPTO_KEY key)
{
	int i, length = 0;
	
	i = log10f((float)key.type);
	length += (int)((i < 0 ? 0 : i) + 1); // key type
	length++; // space
	i = log10f((float)key.num_param);
	length += (int)((i < 0 ? 0 : i) + 1); // no. of key parameters
	length++; // space (or null-terminator for empty key)

	for (i=0; i<key.num_param; i++) {
		length += mpz_sizeinbase(key.param[i], 10) + 1; // key parameter[i]
		length++; // space (or null-terminator)
	}

	return length;
}

int CRYPTO_GetKeyPairStringLength(CRYPTO_KEY_PAIR key_pair)
{
	int i, length = 0;

	length += (int)(log10f((float)key_pair.type) + 1); // key type
	length++; // space

	length += (int)(log10f((float)key_pair.public_key.type) + 1); // public key type
	length++; // space
	length += (int)(log10f((float)key_pair.public_key.num_param) + 1); // no. of key parameters
	length++; // space (or null-terminator for empty key)

	for (i=0; i<key_pair.public_key.num_param; i++) {
		length += mpz_sizeinbase(key_pair.public_key.param[i], 10) + 1; // public key parameter[i]
		length++; // space
	}

	length += (int)(log10f((float)key_pair.private_key.type) + 1); // private key type
	length++; // space
	length += (int)(log10f((float)key_pair.private_key.num_param) + 1); // no. of key parameters
	length++; // space (or null-terminator for empty key)

	for (i=0; i<key_pair.private_key.num_param; i++) {
		length += mpz_sizeinbase(key_pair.private_key.param[i], 10) + 1; // private key parameter[i]
		length++; // space (or null-terminator)
	}
	
	return length;
}

BOOL CRYPTO_ParseKeyFromString(CRYPTO_KEY *key, char *buffer)
{
	int ptr, i, type, num_param, num_space = 0;

	// check validity of the key string
	if (strstr(buffer, " ") == NULL)
		return FALSE;

	sscanf(buffer, "%d %d", &type, &num_param);
	
	if (type > CRYPTO_KEY_MAX_VALID)
		return FALSE;

	for (i=strlen(buffer) - 1; i>=0; i--)
		if (buffer[i] == ' ')
			num_space++;

	if (num_space < num_param + 1)
		return FALSE;
		
	// release previous allocation
	if (key->param) {
		for (i=0; i<key->num_param; i++) {
			mpz_clear(key->param[i]);
		}
	}

	// reallocate to the new size
	key->param = (mpz_t*)realloc(key->param, sizeof(mpz_t) * num_param);
	//key->param = (mpz_t*)malloc(sizeof(mpz_t) * num_param);
	if (key->param == NULL && key->num_param != 0) {
		return FALSE;
	}

	// initialize the newly allocated mpz_t struct
	for (i=0; i<num_param; i++) {
		mpz_init(key->param[i]);
	}

	// parse the key from the string
	ptr = 0;
	while (buffer[ptr++] != ' '); // skip key type
	while (buffer[ptr++] != ' '); // skip key num parameter

	for (i=0; i<num_param; i++) {
		gmp_sscanf(buffer+ptr, "%Zd", &key->param[i]);
		if (i != num_param-1) {
			while (buffer[ptr++] != ' '); // skip one parameter value
		}
	}
	
	key->type = type;
	key->num_param = num_param;

	return TRUE;
}

BOOL CRYPTO_ParseKeyPairFromString(CRYPTO_KEY_PAIR *key_pair, char *buffer)
{
	int i, ptr, type;

	// check validity of the key string
	if (strstr(buffer, " ") == NULL)
		return FALSE;

	sscanf(buffer, "%d", &type);

	if (type == CRYPTO_KEY_UNDEFINED || type > CRYPTO_KEY_MAX_VALID)
		return FALSE;

	ptr = 0;
	while (buffer[ptr++] != ' ');
	
	//fprintf(stderr, "[DEBUG] buffer = %s\n\n", buffer+ptr);
	
	// parse the public key
	if (CRYPTO_ParseKeyFromString(&key_pair->public_key, buffer+ptr) == FALSE)
		return FALSE;
		
	for (i=0; i<key_pair->public_key.num_param + 2; i++)
		while (buffer[ptr++] != ' ');
		
	//fprintf(stderr, "[DEBUG] buffer = %s\n\n", buffer+ptr);
		
	// parse the private key
	if (CRYPTO_ParseKeyFromString(&key_pair->private_key, buffer+ptr) == FALSE)
		return FALSE;
		
	key_pair->type = type;
	
	return TRUE;
}

BOOL CRYPTO_Encrypt(char cipher[], int *out_len, char data[], int in_len, CRYPTO_KEY key)
{
	if (key.type == CRYPTO_KEY_RSA) {
		return RSA_Encrypt(cipher, out_len, data, in_len, key);
	}
	else {	
		return FALSE;
	}
}

BOOL CRYPTO_Decrypt(char data[], int *out_len, char cipher[], int in_len, CRYPTO_KEY key)
{
	if (key.type == CRYPTO_KEY_RSA) {
		return RSA_Decrypt(data, out_len, cipher, in_len, key);
	}
	else {	
		return FALSE;
	}
}

BOOL CRYPTO_SignPublicKey(char cipher[], int *out_len, CRYPTO_KEY_PAIR pair)
{
	if (pair.type == CRYPTO_KEY_RSA) {
		return RSA_SignPublicKey(cipher, out_len, pair);
	}
	else {
		return FALSE;
	}
}

BOOL CRYPTO_CheckSignedPublicKey(char cipher[], int in_len, CRYPTO_KEY key)
{
	if (key.type == CRYPTO_KEY_RSA) {
		return RSA_CheckSignedPublicKey(cipher, in_len, key);
	}
	else {
		return FALSE;
	}
}

BOOL CRYPTO_CompareKey(CRYPTO_KEY key1, CRYPTO_KEY key2)
{
	int i;
	
	if (key1.type != key2.type)
		return FALSE;
		
	if (key1.num_param != key2.num_param)
		return FALSE;
		
	for (i=0; i<key1.num_param; i++)
		if (mpz_cmp(key1.param[i], key2.param[i]) != 0)
			return FALSE;
	
	return TRUE;
}


unsigned int CRYPTO_GetPlaintextUnitSize(CRYPTO_KEY key)
{
	if (key.type == CRYPTO_KEY_RSA) {
		return RSA_GetPlaintextUnitSize(key);
	}
	else {
		return 0;
	}
}

unsigned int CRYPTO_GetCipherUnitSize(CRYPTO_KEY key)
{
	if (key.type == CRYPTO_KEY_RSA) {
		return RSA_GetCipherUnitSize(key);
	}
	else {
		return 0;
	}
}


void CRYPTO_AddKeyToListFile(CRYPTO_KEY key, char user_name[], char filename[])
{
	FILE *fp;
	char *key_str = NULL;
	
	// key to string
	CRYPTO_KeyToStringAlloc(&key_str, key);
	
	// append the file
	fp = fopen(filename, "at");
	
	// add entry
	if (fp && key_str) {
		fprintf(fp, "%s %s\n", user_name, key_str);
	}
	
	// release resource
	if (fp)
		fclose(fp);
		
	if (key_str)
		free(key_str);
}


BOOL CRYPTO_QueryUserFromListFile(char user_name[], int name_buf_size, CRYPTO_KEY key, char filename[])
{
	FILE *fp = NULL;
	char *buffer, *user, *ptr, *key_str = NULL;
	int i;
	
	buffer = (char*) malloc(sizeof(char) * 4096);
	user = (char*) malloc(sizeof(char) * 4096);
	key_str = (char*) malloc(sizeof(char) * 4096);
	
	CRYPTO_KeyToStringAlloc(&key_str, key);
	
	// open key list file
	fp = fopen(filename, "rt");
	
	// check each entry in the list file
	if (fp && key_str) {
		while (!feof(fp)) {
			fgets(buffer, 4096, fp);
			if (buffer[strlen(buffer)-1] == '\n')
				buffer[strlen(buffer)-1] = 0;
			
			for (i=0; buffer[i] != ' ' && i < 4096; i++)
				user[i] = buffer[i];
			user[i] = 0;
				
			ptr = &buffer[i+1];
			
			if (strcmp(key_str, ptr) == 0) {
				memcpy(user_name, user, name_buf_size);
				user_name[name_buf_size-1] = 0;
				return TRUE;
			}
		}
		
		fclose(fp);
	}
	
	// release resource
	if (buffer)
		free(buffer);
	if (user)
		free(user);
	if (key_str)
		free(key_str);
		
	return FALSE;
}


void CRYPTO_RemoveKeyFromListFile(CRYPTO_KEY key, char filename[])
{
}

void CRYPTO_RemoveUserFromListFile(char user_name[], char filename[])
{
}
